<?php

namespace App\Services;

use App\Enums\EmailAndTelephoneTypeEnum;
use App\Exceptions\InvalidDeletingPrimaryTelephoneAttempt;
use App\Exceptions\InvalidEditingPrimaryTelephoneAttempt;
use App\Exceptions\InvalidNumberDeactivationAttempt;
use App\Exceptions\InvalidPasswordException;
use App\Exceptions\InvalidTelephone;
use App\Exceptions\InvalidTPCodeException;
use App\Exceptions\TelephoneAlreadyExistException;
use App\Models\OtpVerification;
use App\Models\User;
use App\Models\UserTelephone;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait UserTelephoneService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return UserTelephone::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Check if telephone number already exist
    *
    * @param string $telephone
    * @return bool
    */
   public static function checkNumberExist(string $telephone): bool
   {
      return UserTelephone::query()->where('telephone', $telephone)->exists();
   }

   /**
    * Find by telephone number
    *
    * @param int $userId
    * @param string $telephone
    * @return Model|Builder|null
    */
   public static function findByUserTelephone(int $userId, string $telephone): Model|Builder|null
   {
      return UserTelephone::query()->where('user_id', $userId)->where('telephone', $telephone)->first();
   }

   /**
    * Check if user already has primary telephone
    *
    * @param int $userId
    * @return bool
    */
   public static function alreadyHasPrimaryTelephone(int $userId): bool
   {
      return UserTelephone::query()->where('user_id', $userId)->where('type', EmailAndTelephoneTypeEnum::primary->value)->exists();
   }

   /**
    * Add telephone
    *
    * @param array $data
    * @return Builder|Model
    * @throws InvalidPasswordException
    * @throws TelephoneAlreadyExistException
    */
   public static function addTelephone(array $data): Model|Builder
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();

      try {
         // Check phone number is unique and is not already in use by the current user
         User::checkPhoneNumberIsUniqueAndNotUsedByCurrentUser($data["telephone"]);

         // Check if user already has a primary telephone
         $alreadyHasPrimaryTelephone = self::alreadyHasPrimaryTelephone($user->{"id"});

         if (!$alreadyHasPrimaryTelephone) {
            // Update user phone number
            $user->updateService([
               "telephone" => $data["telephone"],
            ]);
         }

         $telephone = self::findByUserTelephone($user->{"id"}, $data["telephone"]);
         if($telephone !== null) {
            $telephone = $telephone->updateService([
               "type" => $alreadyHasPrimaryTelephone ? EmailAndTelephoneTypeEnum::secondary->value : EmailAndTelephoneTypeEnum::primary->value,
               "verified_at" => now()
            ]);
         } else {
            $telephone = self::store([
               "telephone" => $data["telephone"],
               "type" => $alreadyHasPrimaryTelephone ? EmailAndTelephoneTypeEnum::secondary->value : EmailAndTelephoneTypeEnum::primary->value,
               "user_id" => $user->{"id"},
               "verified_at" => now()
            ]);
         }

         DB::commit();
         $result = $telephone->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserTelephoneService::addTelephone');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Edit telephone number
    *
    * @throws InvalidEditingPrimaryTelephoneAttempt
    * @throws InvalidTelephone
    * @throws TelephoneAlreadyExistException
    * @throws InvalidPasswordException
    */
   public function editTelephoneNumber(array $data): Model|Builder|UserTelephone
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      try {
         $numberToEdit = $this;

         // Check phone number is unique and is not already in use by the current user
         User::checkPhoneNumberIsUniqueAndNotUsedByCurrentUser($data["new_telephone"]);

         if ($numberToEdit->{"type"} === EmailAndTelephoneTypeEnum::primary->value) {
            throw new InvalidEditingPrimaryTelephoneAttempt(__("errors.invalid_primary_telephone_edit_attempt"));
         }

         /** @var UserTelephone $numberToEdit */
         $numberToEdit->updateService([
            "telephone" => $data["new_telephone"],
         ]);

         $results = $numberToEdit->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserTelephoneService::editTelephoneNumber');

         throw $exception;
      }

      return $results;
   }

   /**
    * Mark telephone number as primary
    *
    * @param array $data
    * @return UserTelephone
    * @throws InvalidPasswordException
    * @throws InvalidTelephone
    */
   public function markTelephoneAsPrimary(array $data): UserTelephone
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();

      try {
         $newPrimaryTelephone = $this;

         if ($user->{"userPrimaryNumber"}->{"telephone"} == $data["telephone"]) {
            throw new InvalidTelephone(__("errors.number_already_primary_number"));
         }

         // Update old user primary number to secondary number
         $user->{"userPrimaryNumber"}->updateService([
            "type" => EmailAndTelephoneTypeEnum::secondary->value,
         ]);

         // Update request number to primary number
         $newPrimaryTelephone->updateService([
            "type" => EmailAndTelephoneTypeEnum::primary->value,
         ]);

         // Update user phone number in user table
         $user->updateService([
            "telephone" => $data["telephone"],
         ]);

         DB::commit();
         $result = $newPrimaryTelephone->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserTelephoneService::markTelephoneAsPrimary');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Deactivate number
    *
    * @param array $data
    * @return User|null
    * @throws InvalidNumberDeactivationAttempt
    * @throws InvalidPasswordException
    */
   public function deactivate(array $data): ?User
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      try {
         $numberToDeactivate = $this;

         // Check number is not secondary number
         if ($numberToDeactivate->{"type"} == EmailAndTelephoneTypeEnum::secondary->value) {
            throw new InvalidNumberDeactivationAttempt(__("errors.invalid_number_deactivation_attempt"));
         }

         // Check user has primary email
         if ($user->{"userPrimaryEmail"} == null) {
            throw new InvalidNumberDeactivationAttempt(__("errors.must_have_primary_email_before_deactivate_number"));
         }

         // Deactivate number
         $numberToDeactivate->updateService([
            "deactivated_at" => now()
         ]);

         $result = $user->fresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserTelephoneService::deactivate');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Delete phone number
    *
    * @param array $data
    * @return Builder|Model
    * @throws InvalidDeletingPrimaryTelephoneAttempt
    * @throws InvalidPasswordException
    * @throws InvalidTelephone
    */
   public function deleteTelephone(array $data): Model|Builder
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      try {
         $telephoneToDelete = $this;

         if ($telephoneToDelete->{"type"} == EmailAndTelephoneTypeEnum::primary->value) {
            throw new InvalidDeletingPrimaryTelephoneAttempt(__("errors.invalid_primary_telephone_delete_attempt"));
         }

         $telephoneToDelete->delete();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserTelephoneService::deleteTelephone');
         throw $exception;
      }

      return $telephoneToDelete->refresh();
   }

   /**
    * Check if number exist as secondary number
    *
    * @param string $telephone
    * @return bool
    */
   public static function checkIsUsedAsSecondaryNumber(string $telephone): bool
   {
      return
         UserTelephone::query()
            ->where("telephone", $telephone)
            ->where('type', EmailAndTelephoneTypeEnum::secondary->value)
            ->exists();
   }

   /**
    * Verify telephone
    *
    * @param array $data
    * @return null
    * @throws InvalidTPCodeException
    */
   public static function verify(array $data): null
   {
      $result = null;

      DB::beginTransaction();

      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone($data['telephone'], null);

         if ($otpEntry === null || $otpEntry->{'otp'} !== $data['otp']) {
            throw new InvalidTPCodeException(__('errors.invalid_otp_code'));
         }

         if ($otpEntry->expired()) {
            throw new InvalidTPCodeException(__('errors.expired_otp_code'));
         }

         // Update otp verification
         $otpEntry->markedAsValidate();

         /** @var UserTelephone $telephoneToVerify */
         $telephoneToVerify = UserTelephone::findByUserTelephone($user->{"id"}, $data["telephone"]);
         if($telephoneToVerify !== null) {
            $telephoneToVerify->updateService([
               'verified_at' => now(),
            ]);
         } else {
            $telephoneToVerify = self::store([
               'telephone' => $data["telephone"],
               'verified_at' => now(),
               'user_id' => $user->{"id"},
               "type" => EmailAndTelephoneTypeEnum::secondary->value,
            ]);
         }

         DB::commit();
         $result = $telephoneToVerify->refresh();
      } catch (Exception $exception) {
         log_debug($exception, 'UserTelephoneService::verify');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }
}
