<?php

namespace App\Services;

use App\Exceptions\InvalidPasswordException;
use App\Exceptions\InvalidTPCodeException;
use App\Models\OtpVerification;
use App\Models\User;
use App\Models\UserEmail;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait UserEmailService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return UserEmail::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Verify email
    *
    * @return mixed
    */
   public function verifyEmail(): mixed
   {
      return $this->updateService([
         "verified_at" => now()
      ]);
   }

   /**
    * Check if telephone number already exist
    *
    * @param string $email
    * @return bool
    */
   public static function checkEmailExist(string $email): bool
   {
      return UserEmail::query()->where('email', $email)->exists();
   }

   /**
    * Find by user email
    *
    * @param int $userId
    * @param string $telephone
    * @return Model|Builder|null
    */
   public static function findByUserEmail(int $userId, string $telephone): Model|Builder|null
   {
      return UserEmail::query()->where('user_id', $userId)->where('email', $telephone)->first();
   }

   /**
    * Add user email
    *
    * @param array $data
    * @return Model|Builder
    * @throws InvalidPasswordException
    * @throws Exception
    */
   public static function addEmail(array $data): Model|Builder
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();

      try {
         /** @var UserEmail $email */
         $email = UserEmail::findByUserEmail($user->{"id"}, $data["email"]);
         if($email !== null) {
            $email->updateService([
               'verified_at' => now(),
            ]);
         } else {
            $email = self::store([
               'email' => $data["email"],
               'verified_at' => now(),
               'user_id' => $user->{"id"},
            ]);
         }

         DB::commit();
         $result = $email->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailService::addEmail');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Edit email
    *
    * @param array $data
    * @return Model|UserEmail|Builder
    * @throws InvalidPasswordException
    * @throws Exception
    */
   public function editEmail(array $data): Model|UserEmail|Builder
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      try {
         $emailToEdit = $this;

         $emailToEdit->updateService([
            "email" => $data["new_email"],
         ]);

         $result = $emailToEdit->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailService::editEmail');

         throw $exception;
      }

      return $result;
   }

   /**
    * Verify email
    *
    * @param array $data
    * @return null
    * @throws InvalidTPCodeException
    */
   public static function verify(array $data): null
   {
      $result = null;

      DB::beginTransaction();

      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone(null, $data['email']);

         if ($otpEntry === null || $otpEntry->{'otp'} !== $data['otp']) {
            throw new InvalidTPCodeException(__('errors.invalid_otp_code'));
         }

         if ($otpEntry->expired()) {
            throw new InvalidTPCodeException(__('errors.expired_otp_code'));
         }

         // Update otp verification
         $otpEntry->markedAsValidate();

         /** @var UserEmail $emailToVerify */
         $emailToVerify = UserEmail::findByUserEmail($user->{"id"}, $data["email"]);
         if($emailToVerify !== null) {
            $emailToVerify->updateService([
               'verified_at' => now(),
            ]);
         } else {
            $emailToVerify = self::store([
               'email' => $data["email"],
               'verified_at' => now(),
               'user_id' => $user->{"id"},
            ]);
         }

         DB::commit();
         $result = $emailToVerify->refresh();
      } catch (Exception $exception) {
         log_debug($exception, 'UserEmailService::verify');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Delete email
    *
    * @throws void
    * @throws InvalidPasswordException
    * @throws Exception
    */
   public function deleteEmail(array $data): void
   {
      /** @var User $user */
      $user = auth()->user();

      // Verify user password
      $user->checkUserPassword($data["password"]);

      try {
         $this->delete();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailService::deleteEmail');

         throw $exception;
      }
   }
}
