<?php

namespace App\Services;

use App\Enums\DocumentTypeEnum;
use App\Models\SeasonEpisode;
use App\Models\Series;
use App\Models\SeriesSeason;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

trait SeasonEpisodeService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SeasonEpisode::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Add new season episode
     *
     * @param string $filePath
     * @param array $data
     * @param SeriesSeason $season
     * @return Model|Builder
     */
    public static function createNewEpisode(SeriesSeason $season, string $filePath, array $data): Model|Builder
    {
        return self::store([
            'designation' => $data['episode_season_title'] ?? null,
            'description' => $data['episode_season_description'] ?? null,
            'occurrence' => $data['episode_season_occurrence'],
            'file_path' => $filePath,
            'series_season_id' => $season->{'id'},
        ]);
    }

    /**
     * Add new episode into series season
     *
     * @param SeriesSeason $season
     * @param UploadedFile $episodeFile
     * @param array $data
     * @return Model|Builder|null
     * @throws \Exception
     */
    public static function addNewEpisodeForSeason(SeriesSeason $season, UploadedFile $episodeFile, array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            $product = $season->{'series'}->{'product'};
            $business = $product->{'business'};

            // And add episode ...S
            $basePath = $business->buildProductDir($product->{'id'});
            $basePath = sprintf(
                config('torryme.paths.docs'),
                sprintf(
                    $basePath . "/" . SeriesSeason::$prefixDir . "%s",
                    $season->{'occurrence'}
                ),
            );

            // Try to store episode ...
            $resultPath = upload_file_system(
                DocumentTypeEnum::seriesEpisodeFile->value,
                $basePath,
                $episodeFile,
            );

            if (filled($resultPath)) {
                $episode = self::createNewEpisode($season, $resultPath, $data);

                // Try to update product status
                if (!$product->isComplete()) {
                    $product->updateStatusToComplete();
                }

                DB::commit();
                return $episode->refresh();
            }

            DB::rollBack();
        } catch (\Exception $exception) {
            DB::rollBack();
            log_debug(exception: $exception, prefix: 'SeasonEpisodeService::addNewEpisodeForSeason');

            throw $exception;
        }

        return null;
    }

    /**
     * Total number of episodes in the series
     *
     * @param Series $series
     * @return int
     */
    public static function totalSeriesEpisode(Series $series): int
    {
        return SeasonEpisode::query()
            ->whereHas('seriesSeason.series', function (Builder $builder) use ($series) {
                $builder->where('id', $series->{'id'});
            })
            ->count();
    }

    /**
     * Total number of episodes in the series season
     *
     * @param SeriesSeason $season
     * @return int
     */
    public static function totalSeriesSeasonEpisode(SeriesSeason $season): int
    {
        return SeasonEpisode::query()
            ->whereHas('seriesSeason', function (Builder $builder) use ($season) {
                $builder->where('id', $season->{'id'});
            })
            ->count();
    }

    /**
     * Compute sample season episode attributes
     *
     * @param Collection $episodes
     * @return array
     */
    public static function computeSampleDetails(Collection $episodes): array
    {
        $result = [];

        foreach ($episodes as $episode) {
            $result[] = array(
                'occurrence' => $episode->{'occurrence'},
                'designation' => $episode->{'designation'},
                'episode_url' => $episode->{'episode_url'},
                'size' => $episode->episodeSize(),
                'duration' => $episode->episodeDuration(),
            );
        }

        return $result;
    }
}
