<?php

namespace App\Services;

use App\Enums\AttachmentTypeEnum;
use App\Models\RealtimeMessageAttachment;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

trait RealtimeMessageAttachmentService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return RealtimeMessageAttachment::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Find by uuid
    *
    * @param string $uuid
    * @return Builder|Model|null
    */
   public static function findByUuid(string $uuid): Model|Builder|null
   {
      return RealtimeMessageAttachment::query()->where('uuid', $uuid)->first();
   }

   /**
    * Find by id
    *
    * @param int $id
    * @return Model|Builder|null
    */
   public static function findById(int $id): Model|Builder|null
   {
      return RealtimeMessageAttachment::query()->find($id);
   }

   /**
    * Serialize for channel interaction
    *
    * @return array
    */
   public function serializeForChannel(): array
   {
      $attachment = $this;
      $serializedData = array(
         'uuid' => $attachment->{'uuid'},
         'url' => $attachment->buildAttachmentUrl(),
         'remote_name' => $attachment->{'path'},
         'original_name' => $attachment->{'original_name'},
         'attachment_type' => $attachment->{'attachment_type'},
      );

      // Meta data ...
      $metaData = file_metadata($attachment->buildAttachmentFilePath());
      $metaData['size'] = $attachment->mediaSize();
      $serializedData['metadata'] = $metaData;

      // Thumbnail ...
      if(in_array($attachment->{'attachment_type'}, [AttachmentTypeEnum::video->{'value'}, AttachmentTypeEnum::image->{'value'}])) {
         $serializedData['thumbnail_url'] = $attachment->buildAttachmentThumbnailUrl();
      }

      return $serializedData;
   }
}
