<?php

namespace App\Services;

use App\Concerns\ProductShopConcern;
use App\Enums\GameTypeEnum;
use App\Enums\GameUploadKeyEnum;
use App\Enums\MusicTypeEnum;
use App\Enums\MusicUploadKeyEnum;
use App\Enums\OrderStatusEnum;
use App\Enums\ProductStatusEnum;
use App\Enums\SeriesUploadKeyEnum;
use App\Exceptions\UploadFileException;
use App\Models\Business;
use App\Models\Currency;
use App\Models\ExchangeRate;
use App\Models\Game;
use App\Models\HashTag;
use App\Models\Like;
use App\Models\MusicAlbum;
use App\Models\Order;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductHashTag;
use App\Models\ProductRating;
use App\Models\Save;
use App\Models\Series;
use App\Models\User;
use App\Models\Wallet;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait ProductService
{
    use ProductShopConcern;

    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Product::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $productId
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $productId): Model|Collection|Builder|array|null
    {
        return Product::query()->find($productId);
    }

    /**
     * Create Goods product
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function createGoodsProduct(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $businessId = $user->{"business"}->{"id"};
            $businessCategoryId = $user->{"business"}->{"business_category_id"};
            $priceCurrency = Currency::findByCode($data["price_currency"]);
            $quantity = $data["quantity"] ?? null;
            $unit = $data["unit"] ?? null;

            // Create goods product
            $product = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => $quantity === null && $unit === null ? config('torryme.constants.default_quantity') : $quantity,
                "unit" => $quantity === null && $unit === null ? null : $unit,
                "price" => $data["price"],
                "price_currency_id" => $priceCurrency != null ? $priceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => $data["min_order"] ?? config('torryme.constants.default_min_order'),
                "max_order" => $data["max_order"] ?? config('torryme.constants.default_max_order'),
                "creation_status" => ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $businessCategoryId,
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            self::handleHashTag($data, $product->{"id"}, $businessId);
            self::handleCategoryAttributes($data, $product->{"id"}, $wallet->{"currency_id"});

            DB::commit();
            $result = $product;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::createGoodsProduct');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Create service product
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function createServiceProduct(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $businessId = $user->{"business"}->{"id"};
            $businessCategoryId = $user->{"business"}->{"business_category_id"};
            $servicePriceCurrency = Currency::findByCode($data["price_currency"]);

            // Create service product
            $serviceProduct = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => $data["quantity"],
                "delivery_in_days" => $data["delivery_in_days"],
                "price" => $data["price"],
                "price_currency_id" => $servicePriceCurrency != null ? $servicePriceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => $data["min_order"] ?? config('torryme.constants.default_min_order'),
                "max_order" => $data["max_order"] ?? config('torryme.constants.default_max_order'),
                "creation_status" => ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $businessCategoryId,
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            self::handleHashTag($data, $serviceProduct->{"id"}, $businessId);
            self::handleCategoryAttributes($data, $serviceProduct->{"id"}, $wallet->{"currency_id"});

            DB::commit();
            $result = $serviceProduct;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::createServiceProduct');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Create cinema product
     *
     * @param array $data
     * @return Model|Builder|null
     * @throws UploadFileException
     */
    public static function createCinemaProduct(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $business = $user->{"business"};
            $businessId = $user->{"business"}->{"id"};
            $businessCategoryId = $user->{"business"}->{"business_category_id"};
            $cinemaPriceCurrency = Currency::findByCode($data["price_currency"]);

            // Create service product
            /** @var Product $cinemaProduct */
            $cinemaProduct = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => config('torryme.constants.default_quantity'),
                "price" => $data["price"] ?? config('torryme.constants.default_price'),
                "price_currency_id" => $cinemaPriceCurrency != null ? $cinemaPriceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => config('torryme.constants.default_min_order'),
                "max_order" => config('torryme.constants.default_max_order'),
                "creation_status" => ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $businessCategoryId,
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            self::handleHashTag($data, $cinemaProduct->{"id"}, $businessId);
            self::handleCategoryAttributes($data, $cinemaProduct->{"id"}, $wallet->{"currency_id"});

            if ($cinemaProduct->{'businessSubCategoryChild'}->{'code'} === config('torryme.business_sub_category_child_codes.series')) {
                $series = Series::createSeries(
                    $data[SeriesUploadKeyEnum::seriesPoster->value],
                    $cinemaProduct,
                    $business
                );

                $cinemaProduct->updateService([
                    "series_id" => $series->{"id"},
                ]);
            }

            DB::commit();
            $result = $cinemaProduct->refresh();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::createCinemaProduct');
            DB::rollBack();

            throw $exception;
        }

        return $result;
    }

    /**
     * Create music product
     *
     * @param array $data
     * @return Builder|Model
     * @throws UploadFileException
     */
    public static function createMusicProduct(array $data): Model|Builder
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $business = $user->{"business"};
            $businessId = $user->{"business"}->{"id"};
            $cinemaPriceCurrency = Currency::findByCode($data["price_currency"]);

            // Create music product
            /** @var Product $musicProduct */
            $musicProduct = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => config('torryme.constants.default_quantity'),
                "price" => $data["price"],
                "price_currency_id" => $cinemaPriceCurrency != null ? $cinemaPriceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => config('torryme.constants.default_min_order'),
                "max_order" => config('torryme.constants.default_max_order'),
                "creation_status" => ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $user->{"business"}->{"business_category_id"},
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            self::handleHashTag($data, $musicProduct->{"id"}, $user->{"business"}->{"id"});
            self::handleCategoryAttributes($data, $musicProduct->{"id"}, $wallet->{"currency_id"});

            if ($data["music_type"] === MusicTypeEnum::album->value) {
                $musicAlbum = MusicAlbum::createMusicAlbum(
                    $data[MusicUploadKeyEnum::albumPoster->value],
                    $musicProduct,
                    $business
                );

                $musicProduct->updateService([
                    "music_album_id" => $musicAlbum->{"id"},
                ]);
            }

            DB::commit();
            $result = $musicProduct;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::createMusicProduct');
            DB::rollBack();

            throw $exception;
        }

        return $result;
    }

    /**
     * Create library product
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function createLibraryProduct(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $businessId = $user->{"business"}->{"id"};
            $businessCategoryId = $user->{"business"}->{"business_category_id"};
            $libraryPriceCurrency = Currency::findByCode($data["price_currency"]);

            // Create library product
            /** @var Product $libraryProduct */
            $libraryProduct = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => isset($data["quantity"]) && $data["quantity"] > config("torryme.constants.default_zero_number") ? $data["quantity"] : config('torryme.constants.default_quantity'),
                "price" => $data["price"] ?? config('torryme.constants.default_price'),
                "price_currency_id" => $libraryPriceCurrency != null ? $libraryPriceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => isset($data["min_order"]) && $data["min_order"] > config("torryme.constants.default_zero_number") ? $data["min_order"] : config('torryme.constants.default_min_order'),
                "max_order" => isset($data["max_order"]) && $data["max_order"] >= config("torryme.constants.default_zero_number") ? $data["max_order"] : config('torryme.constants.default_max_order'),
                "creation_status" => ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $businessCategoryId,
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            self::handleHashTag($data, $libraryProduct->{"id"}, $businessId);
            self::handleCategoryAttributes($data, $libraryProduct->{"id"}, $wallet->{"currency_id"});

            DB::commit();
            $result = $libraryProduct;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::createLibraryProduct');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Add game product
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function addGameProduct(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $wallet = $user->{"wallet"};
            $businessId = $user->{"business"}->{"id"};
            $businessCategoryId = $user->{"business"}->{"business_category_id"};
            $priceCurrency = Currency::findByCode($data["price_currency"]);

            // Create game product
            /** @var Product $product */
            $product = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "quantity" => $data["quantity"] ?? config('torryme.constants.default_quantity'),
                "price" => $data["price"],
                "price_currency_id" => $priceCurrency != null ? $priceCurrency->{"id"} : $wallet->{"currency_id"},
                "min_order" => $data["min_order"] ?? config('torryme.constants.default_min_order'),
                "max_order" => $data["max_order"] ?? config('torryme.constants.default_max_order'),
                "creation_status" => $data['game_type'] === GameTypeEnum::hardCopy->value ? ProductStatusEnum::complete->value : ProductStatusEnum::incomplete->value,
                "business_id" => $businessId,
                "business_category_id" => $businessCategoryId,
                "business_sub_category_id" => $data["business_sub_category_id"],
                "business_sub_category_child_id" => $data["business_sub_category_child_id"],
                "business_sub_category_grand_child_id" => $data["business_sub_category_grand_child_id"] ?? null,
            ]);

            if ($data['game_type'] === GameTypeEnum::hardCopy->value) {
                $basePath = sprintf(config('torryme.paths.docs'), $user->{"business"}->buildProductDir($product->{'id'}));

                // Try to upload file ...
                $resultPath = upload_file_system(
                    GameUploadKeyEnum::gameCover->value,
                    $basePath,
                    $data[GameUploadKeyEnum::gameCover->value],
                );

                if (!filled($resultPath)) {
                    throw new UploadFileException();
                }

                $game = Game::createNewGame($product);

                $product->updateService([
                    "game_id" => $game->{"id"},
                ]);
            };

            if ($data['game_type'] === GameTypeEnum::softCopy->value) {
                // ... Nothing to do
            };

            self::handleHashTag($data, $product->{"id"}, $businessId);
            self::handleCategoryAttributes($data, $product->{"id"}, $wallet->{"currency_id"});

            DB::commit();
            $result = $product;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductService::addGameProduct');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Update create status
     *
     * @return mixed
     */
    public function updateStatusToComplete(): mixed
    {
        return $this->updateService([
            "creation_status" => ProductStatusEnum::complete->value,
        ]);
    }

    /**
     * Is complete
     *
     * @return bool
     */
    public function isComplete(): bool
    {
        return $this->{'creation_status'} === ProductStatusEnum::complete->value;
    }

    /**
     * Get all product hashtags as string
     *
     * @return array
     */
    public function getHashTagsAsString(): array
    {
        return $this->tags()->with('hashTag')->get()->pluck('hashTag.designation')->toArray();
    }

    /**
     * Get all attachments urls
     *
     * @return array
     */
    public function allAttachmentUrls(): array
    {
        return array_map(function ($attachment) {
            return $attachment['attachment_url'];
        }, $this->{'productAttachments'}->toArray());
    }

    /**
     * Check if the user liked the product
     *
     * @param User|null $user
     * @return bool
     */
    public function userLiked(?User $user): bool
    {
        if ($user === null) {
            return false;
        }

        $queryBuilder = Like::query()->where('product_id', $this->{'id'});
        return $queryBuilder->where('user_id', $user->{'id'})->exists();
    }

    /**
     * Check if the user saved the product
     *
     * @param User|null $user
     * @return bool
     */
    public function userSaved(?User $user): bool
    {
        if ($user === null) {
            return false;
        }

        $queryBuilder = Save::query()->where('product_id', $this->{'id'});
        return $queryBuilder->where('user_id', $user->{'id'})->exists();
    }

    /**
     * Get all ratings
     *
     * @param int|null $page
     * @return Collection|LengthAwarePaginator|array
     */
    public function allRatings(int $page = null): Collection|LengthAwarePaginator|array
    {
        return ProductRating::allProductRatings($this, $page);
    }

    /**
     * Total rating
     *
     * @return int
     */
    public function totalRating(): int
    {
        return ProductRating::totalProductRating($this);
    }

    /**
     * Average rating
     *
     * @return mixed
     */
    public function averageRating(): mixed
    {
        return ProductRating::computeAverageRating($this);
    }

    // Utilities

    /**
     * Handle category hashtags
     *
     * @param array $data
     * @param int $productId
     * @param int $businessId
     */
    public static function handleHashTag(array $data, int $productId, int $businessId): void
    {
        foreach ($data["hash_tags"] as $hashTagKey => $hashTagValue) {
            // Check if hashtag already exist
            $hashTagExist = HashTag::checkExist($hashTagValue);

            // Hashtag does not exist
            if ($hashTagExist == null) {
                $hashTag = HashTag::store([
                    "designation" => $hashTagValue,
                ]);

                // Create product hashtags
                ProductHashTag::store([
                    "product_id" => $productId,
                    "hash_tag_id" => $hashTag->{"id"},
                    "business_id" => $businessId,
                ]);
            } else {
                // Create product hashtags
                ProductHashTag::store([
                    "product_id" => $productId,
                    "hash_tag_id" => $hashTagExist->{"id"},
                    "business_id" => $businessId,
                ]);
            }
        }
    }

    /**
     * Handle category attributes
     *
     * @param array $data
     * @param int $productId
     * @param int $walletCurrency
     */
    public static function handleCategoryAttributes(array $data, int $productId, int $walletCurrency): void
    {
        if (isset($data["category_attributes"])) {
            foreach ($data["category_attributes"] as $categoryAttributes) {
                $additionalPriceCurrency = null;
                $additionalPrice = $categoryAttributes["additional_price"] ?? null;

                if (isset($categoryAttributes["additional_price_currency"])) {
                    $additionalPriceCurrency = Currency::findByCode(
                        $categoryAttributes["additional_price_currency"],
                    );
                }

                if (isset($categoryAttributes["attribute_id"]) || isset($categoryAttributes["attribute_value_id"])) {
                    // Create product attributes
                    ProductAttribute::store([
                        "value" => $categoryAttributes["value"] ?? null,
                        "extra_value" => $categoryAttributes["extra_value"] ?? null,
                        "additional_price" => $additionalPrice,
                        "additional_price_currency_id" => $additionalPrice !== null
                            ? $additionalPriceCurrency != null ? $additionalPriceCurrency->{"id"} : $walletCurrency
                            : null,
                        "product_id" => $productId,
                        "attribute_id" => $categoryAttributes["attribute_id"] ?? null,
                        "attribute_value_id" => $categoryAttributes["attribute_value_id"] ?? null,
                    ]);
                }
            }
        }
    }

    /**
     * Determines whether the user can purchase the product
     *
     * @param User $user
     * @return bool
     */
    public function userCanBuy(User $user): bool
    {
        return $user->{'id'} !== $this->{'business'}->{'owner_user_id'};
    }

    /**
     * Convert product price to auth user currency
     *
     * @return array
     */
    public function convertPriceToAuthUserCurrency(): array
    {
        $product = $this;

        //Get auth user
        /** @var User $user */
        $user = auth()->user();

        /** @var Wallet $authUserWallet */
        $authUserWallet = $user->{"wallet"};

        if ($authUserWallet->isSameCurrency($product->{"price_currency_id"}, $authUserWallet->{"currency_id"})) {
            $convertedPrice = $product->{'price'};
        } else {
            // Get exchange rate for product and connected user currencies
            $exchangeRate = ExchangeRate::getExchangeRate($product->{"price_currency_id"}, $authUserWallet->{"currency_id"});
            // Multiply product price by exchange rate to get corresponding product price in connected user currency
            $convertedPrice = $product->{'price'} * $exchangeRate->{"exchange_rate"};
        }

        return array(
            "price" => $convertedPrice,
            "currency" => $authUserWallet->{"currency"}->{"code"},
        );
    }

    /**
     * Check product has already been purchased
     *
     * @return bool
     */
    public function checkProductAlreadyPurchased(): bool
    {
        // Get auth user
        /** @var User $user */
        $user = auth()->user();

        $productId = $this->{'id'};

        return
            Order::query()
                ->where("customer_user_id", $user->{"id"})
                ->where("status", OrderStatusEnum::completed->value)
                ->whereHas("orderDetails.product", function (Builder $o) use ($productId) {
                    $o->where('id', $productId);
                })->exists();
    }

    /**
     * Get products with ratings
     *
     * @param Business $business
     * @param int|null $page
     * @return LengthAwarePaginator
     */
    public static function getProductsWithRatings(Business $business, int $page = null): LengthAwarePaginator
    {
        $queryBuilder =
            Product::query()
                ->has('productRating')
                ->where("business_id", $business->{"id"})
                ->where("creation_status", ProductStatusEnum::complete->value)
                ->orderByDesc('updated_at');

        return $queryBuilder->paginate(
            perPage: config('torryme.constants.items_per_page'),
            page: $page
        );
    }

}
