<?php

namespace App\Services;

use App\Models\CartItem;
use App\Models\Product;
use App\Models\ProductAttribute;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

trait ProductAttributeService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return ProductAttribute::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Compute attribute for product details
     *
     * @param Product $product
     * @return array
     */
    public static function computeForProductDetails(Product $product): array
    {
        $result = [];
        $productAttributes = $product->{'productAttributes'};

        foreach ($productAttributes as $productAttribute) {
            $attributeValue = $productAttribute->{'attributeValue'};
            $attributeSIUnit = $attributeValue?->{'attributeSIUnit'};
            $linkAttribute = $productAttribute->{'attribute'} ?? $attributeValue->{'attribute'};

            if (!array_key_exists($linkAttribute->{'id'}, $result)) {
                $result[$linkAttribute->{'id'}] = array(
                    'attribute' => [
                        'id' => $linkAttribute->{'id'},
                        'type' => $linkAttribute->{'type'},
                        'designation' => $linkAttribute->{'designation'},
                    ],
                    'attribute_value' => $attributeValue !== null ? [
                        'id' => $attributeValue?->{'id'},
                        'si_unit' => [
                            'id' => $attributeSIUnit->{'id'},
                            'si_unit' => $attributeSIUnit->{'si_unit'},
                            'si_unit_designation' => $attributeSIUnit->{'si_unit_designation'},
                        ]
                    ] : null,
                    'values' => []
                );
            }

            $result[$linkAttribute->{'id'}]['values'][] = array(
                'id' => $productAttribute->{'id'},
                'value' => $productAttribute->{'value'},
                'extra_value' => $productAttribute->{'extra_value'},
                'additional_price' => $productAttribute->{'additional_price'},
                'additional_price_currency' => $productAttribute->{'additionalPriceCurrency'}?->{'code'},
            );
        }

        return array_values($result);
    }

    /**
     * Compute attribute for request details
     *
     * @return array
     */
    public function computeForDetails(): array
    {
        $productAttribute = $this;

        $attributeValue = $productAttribute->{'attributeValue'};
        $attributeSIUnit = $attributeValue?->{'attributeSIUnit'};
        $linkAttribute = $productAttribute->{'attribute'} ?? $attributeValue->{'attribute'};

        return array(
            'attribute' => [
                'id' => $linkAttribute->{'id'},
                'type' => $linkAttribute->{'type'},
                'designation' => $linkAttribute->{'designation'},
            ],
            'attribute_value' => $attributeValue !== null ? [
                'id' => $attributeValue?->{'id'},
                'si_unit' => [
                    'id' => $attributeSIUnit->{'id'},
                    'si_unit' => $attributeSIUnit->{'si_unit'},
                    'si_unit_designation' => $attributeSIUnit->{'si_unit_designation'},
                ]
            ] : null,
            'values' => [
                'id' => $productAttribute->{'id'},
                'value' => $productAttribute->{'value'},
                'extra_value' => $productAttribute->{'extra_value'},
                'additional_price' => $productAttribute->{'additional_price'},
                'additional_price_currency' => $productAttribute->{'additionalPriceCurrency'}?->{'code'},
            ]
        );
    }

    /**
     * CHeck product attribute exist
     *
     * @param int $productAttributeId
     * @return bool
     */
    public static function checkAttributeExist(int $productAttributeId): bool
    {
        return ProductAttribute::query()->where("id", $productAttributeId)->exists();
    }
}
