<?php

namespace App\Services;

use App\Enums\EscrowStatusEnum;
use App\Enums\OrderStatusEnum;
use App\Enums\OrderTypesEnum;
use App\Enums\PaymentGateWayEnum;
use App\Enums\TransactionStatusEnum;
use App\Enums\TransactionTypesNum;
use App\Exceptions\CouponExpiredException;
use App\Exceptions\DoublePurchaseException;
use App\Exceptions\InsufficientWalletBalanceException;
use App\Exceptions\InvalidBookPurchaseException;
use App\Exceptions\InvalidBusinessCoupon;
use App\Exceptions\InvalidCouponAmountToSpendException;
use App\Exceptions\InvalidOrderCancellationAttempt;
use App\Exceptions\InvalidOrderDeliveryConfirmationAttempt;
use App\Exceptions\InvalidOrderDeliveryDate;
use App\Exceptions\InvalidOrderPaymentAttempt;
use App\Exceptions\InvalidOrderValidationAttempt;
use App\Exceptions\WalletBlockedException;
use App\Exceptions\WalletRemainingPinAttemptsException;
use App\Models\Book;
use App\Models\CartItem;
use App\Models\Coupon;
use App\Models\Escrow;
use App\Models\ExchangeRate;
use App\Models\Game;
use App\Models\Movie;
use App\Models\Music;
use App\Models\MusicAlbum;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\OrderDetailAttribute;
use App\Models\Series;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Wallet;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

trait OrderService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return Order::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Find by id
    *
    * @param int $id
    * @return Builder|Builder[]|Collection|Model|null
    */
   public static function findById(int $id): Model|Collection|Builder|array|null
   {
      return Order::query()->find($id);
   }

   /**
    * Check payer is owner of order
    *
    * @throws InvalidOrderPaymentAttempt
    */
   public function checkPayerIsOrderOwner(int $userId): void
   {
      if ($userId !== $this->{"customer_user_id"}) {
         throw  new InvalidOrderPaymentAttempt(__("errors.invalid_order_payment_attempt"));
      }
   }

   /**
    * Check request user is supplier of order
    *
    * @throws InvalidOrderValidationAttempt
    */
   public function isSupplierBusiness($businessId): void
   {
      if ($businessId !== $this->{"supplier_business_id"}) {
         throw  new InvalidOrderValidationAttempt(__("errors.invalid_order_validation_attempt"));
      }
   }

   /**
    * Check user can confirm order
    *
    * @throws InvalidOrderDeliveryConfirmationAttempt
    */
   public function canUserConfirmOrder(): void
   {
      /** @var User $user */
      $user = auth()->user();
      $business = $user->{'business'};

      if ($business != null && $business->{"id"} !== $this->{"supplier_business_id"} && $user->{"id"} !== $this->{"customer_user_id"}) {
         throw  new InvalidOrderDeliveryConfirmationAttempt(__("errors.invalid_order_delivery_confirmation_attempt"));
      }
   }

   /**
    * Check supplier has marked order as delivered
    *
    * @throws InvalidOrderDeliveryConfirmationAttempt
    */
   public function hasSupplierConfirmedDelivery(): void
   {
      if ($this->{"delivered_at"} === null) {
         throw  new InvalidOrderDeliveryConfirmationAttempt(__("errors.invalid_order_delivery_confirmation_attempt"));
      }
   }

   /**
    * Check command can be confirmed
    *
    * @throws InvalidOrderDeliveryConfirmationAttempt
    */
   public function hasBeenPaidSoCanConfirm(): void
   {
      if ($this->{"escrow"} == null) {
         throw new InvalidOrderDeliveryConfirmationAttempt(__("errors.invalid_order_delivery_confirmation_attempt"));
      }
   }

   /**
    * Check user is allowed to cancel order
    *
    * @throws InvalidOrderCancellationAttempt
    */
   public function canUserCancelOrder(): void
   {
      /** @var User $user */
      $user = auth()->user();
      $business = $user->{'business'};

      if ($business != null && $business->{"id"} !== $this->{"supplier_business_id"} && $user->{"id"} !== $this->{"customer_user_id"}) {
         throw new InvalidOrderCancellationAttempt(__("errors.invalid_order_cancellation_attempt"));
      }
   }

   /**
    * Check command is eligible for business  cancellation
    *
    * @throws InvalidOrderCancellationAttempt
    */
   public function eligibleForBusinessCancellation(): void
   {
      if ($this->{"status"} == OrderStatusEnum::completed->value) {
         throw new InvalidOrderCancellationAttempt(__("errors.command_not_eligible_for_cancellation"));
      }
   }

   /**
    * Check if command has been paid for
    *
    * @throws InvalidOrderCancellationAttempt
    */
   public function hasBeenPaidSoCanCancel(): void
   {
      if ($this->{"escrow"} == null) {
         throw new InvalidOrderCancellationAttempt(__("errors.invalid_order_cancellation_attempt"));
      }
   }

   /**
    * Check delivery date is valid
    *
    * @param string $deliveryDate
    * @throws InvalidOrderDeliveryDate
    */
   public static function isValidDeliveryDate(string $deliveryDate): void
   {
      $currentDate = Carbon::now();
      if ($currentDate->gt($deliveryDate)) {
         throw  new  InvalidOrderDeliveryDate(__("errors.invalid_order_delivery_date"));
      }
   }

   /**
    * Check order has shipping cost
    *
    * @return bool
    */
   public function hasShippingCost(): bool
   {
      return $this->{"customer_shipping_amount"} !== null;
   }

   /**
    * Check if order has been cancelled
    *
    * @throws InvalidOrderPaymentAttempt
    */
   public function hasBeenCancelled()
   {
      if ($this->{"cancelled_at"} != null) {
         throw new InvalidOrderPaymentAttempt(__("errors.order_cancelled"));
      }
   }

   /**
    * Add order
    *
    * @param array $data
    * @return Order
    * @throws CouponExpiredException
    * @throws InvalidBusinessCoupon
    * @throws InvalidCouponAmountToSpendException
    */
   public static function addOrder(array $data): Order
   {
      DB::beginTransaction();
      //TODO check user cannot order their own merchandise

      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var Coupon $coupon */

         /** @var Order $order */

         $userWallet = $user->{"wallet"};
         $customerCurrency = $userWallet->{"currency_id"};

         // Get all cart items and their corresponding products, grouped by business id
         $cartItemList = CartItem::getCartItemListByBusiness($data["cart_ids"]);

         foreach ($cartItemList as $listItem) {

            $totalCartPrice = array();
            foreach ($listItem as $item) {
               // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
               /** @var CartItem $cartItem */
               $cartItem = CartItem::findById($item->{"cart_item_id"});
               // Get total price for each cart item
               $totalCartItemPrice = $cartItem->computeCartItemTotalPrice();

               // Join all cart item prices for a business together
               $totalCartPrice[] = $totalCartItemPrice;
            }

            // Sum all cart item prices for a business
            $totalCartAmount = array_sum($totalCartPrice);

            $totalCustomerAmount = self::handlePriceCurrencyConversion($userWallet, $totalCartAmount, $listItem[0]->{"price_currency_id"}, $customerCurrency);

            //In cases where there are no discounts, amount before and amount after discount should be the same
            $totalCustomerAmountAfterCoupon = $totalCustomerAmount;
            $totalSupplierAmountAfterCoupon = $totalCartAmount;

            //Initiate variables
            $totalCustomerProductAmountToExclude = 0;
            $totalSupplierProductAmountToExclude = 0;

            //TODO ensure coupon can only be applied to products belonging to business that created the coupons
            if (isset($data["coupon_code"]) && $data["coupon_code"] != null) {
               $coupon = Coupon::findByCode($data["coupon_code"]);

               //Check if coupon has expired
               $coupon->hasExpired();

               //Check if coupon belong to business product
               $coupon->isValidBusinessCoupon($listItem[0]->{"business_id"});

               // Apply percentage card discount to total order amount
               if ($coupon->isPercentageCartDiscount()) {

                  self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmountAfterCoupon, $customerCurrency);

                  //Handle customer product exclusion
                  $totalCustomerProductAmountToExclude = self::handleCustomerProductExclusion($listItem, $coupon, $userWallet, $listItem[0]->{"price_currency_id"}, $customerCurrency, $user->{"id"});

                  //Handle supplier product exclusion
                  $totalSupplierProductAmountToExclude = self::handleSupplierProductExclusion($listItem, $coupon, $user->{"id"});

                  //TODO ensure $totalCustomerAmountAfterProductExclusion is not negative
                  //Subtract total product exclusion amount from customer original cart amount
                  $totalCustomerAmountAfterProductExclusion = $totalCustomerAmount - $totalCustomerProductAmountToExclude;

                  //TODO ensure $totalSupplierAmountAfterProductExclusion is not negative
                  //Subtract total product amount exclusion amount from supplier original cart amount
                  $totalSupplierAmountAfterProductExclusion = $totalCartAmount - $totalSupplierProductAmountToExclude;

                  //Get customer percentage discount amount of customer amount after product exclusion
                  $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmountAfterProductExclusion;

                  //Deduct coupon amount from customer amount after product exclusion
                  $totalCustomerAmountAfterCoupon = $totalCustomerAmountAfterProductExclusion - $customerPercentageCardDiscountAmount;

                  //Get supplier percentage discount amount of cart amount
                  $suppliedPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalSupplierAmountAfterProductExclusion;

                  //Deduct coupon amount from supplier amount after product exclusion
                  $totalSupplierAmountAfterCoupon = $totalSupplierAmountAfterProductExclusion - $suppliedPercentageCardDiscountAmount;
               }

               // Apply fixed card discount to total order amount
               if ($coupon->isFixedCartDiscount()) {

                  self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmountAfterCoupon, $customerCurrency);

                  //TODO Redo service to address bug

                  //Handle product exclusion
                  $totalCustomerProductAmountToExclude = self::handleCustomerProductExclusion($listItem, $coupon, $userWallet, $listItem[0]->{"price_currency_id"}, $customerCurrency, $user->{"id"});

                  //Handle supplier product exclusion
                  $totalSupplierProductAmountToExclude = self::handleSupplierProductExclusion($listItem, $coupon);

                  //TODO ensure $totalCustomerAmountAfterProductExclusion is not negative
                  //Subtract total product exclusion amount from original cart amount
                  $totalCustomerAmountAfterProductExclusion = $totalCustomerAmount - $totalCustomerProductAmountToExclude;

                  //Convert coupon amount to customer currency
                  $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $coupon->{"fixed_card_discount_amount"}, $customerCurrency);
                  //Deduct coupon amount from customer amount after product exclusion
                  $totalCustomerAmountAfterCoupon = $totalCustomerAmountAfterProductExclusion - $customerCouponAmount;


                  //TODO ensure $totalSupplierAmountAfterProductExclusion is not negative
                  //Subtract total product amount exclusion amount from supplier original cart amount
                  $totalSupplierAmountAfterProductExclusion = $totalCartAmount - $totalSupplierProductAmountToExclude;

                  //Convert coupon amount to supplier currency
                  $supplierCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $coupon->{"fixed_card_discount_amount"}, $listItem[0]->{"price_currency_id"});
                  //Deduct coupon amount from supplier amount after product exclusion
                  $totalSupplierAmountAfterCoupon = $totalSupplierAmountAfterProductExclusion - $supplierCouponAmount;
               }

               //Apply percentage discount to product
               if ($coupon->isProductDiscountByPercentage()) {
                  foreach ($listItem as $item) {
                     // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
                     /** @var CartItem $cartItem */
                     $cartItem = CartItem::findById($item->{"cart_item_id"});

                     if ($coupon->{"product_id"} == $cartItem->{"product"}->{"id"}) {
                        //Get product price
                        $totalProductItemPrice = $cartItem->computeCartItemTotalPrice();

                        //Get product price in customer currency
                        $totalCustomerProductAmount = self::handlePriceCurrencyConversion($userWallet, $totalProductItemPrice, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);

                        //Get customer percentage discount amount of total command
                        $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerProductAmount;

                        //Deduct coupon amount from customer amount
                        $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerPercentageCardDiscountAmount;

                        //Get supplier percentage discount amount of total command
                        $suppliedPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalProductItemPrice;

                        //Deduct coupon amount from supplier amount
                        $totalSupplierAmountAfterCoupon = $totalCartAmount - $suppliedPercentageCardDiscountAmount;
                     }
                  }
               }

               // Apply fix amount discount to product
               if ($coupon->isProductDiscountByFixedAmount()) {
                  foreach ($listItem as $item) {
                     // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
                     /** @var CartItem $cartItem */
                     $cartItem = CartItem::findById($item->{"cart_item_id"});

                     if ($coupon->{"product_id"} == $cartItem->{"product"}->{"id"}) {
                        //Get product discount amount
                        $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

                        //Get product price in supplier currency
                        $totalProductItemPrice = $cartItem->computeCartItemTotalPrice();

                        //Convert coupon amount to supplier currency
                        $supplierCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $listItem[0]->{"price_currency_id"});

                        //TODO ensure value is not negative
                        //Deduct coupon amount from supplier amount
                        $totalSupplierAmountAfterCoupon = $totalCartAmount - $supplierCouponAmount;

                        //Get product price in customer currency
                        $customerAmount = self::handlePriceCurrencyConversion($userWallet, $totalProductItemPrice, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);

                        //Convert coupon amount to customer currency
                        $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);

                        //TODO ensure value is not negative
                        //Deduct coupon amount from customer amount
                        $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerCouponAmount;
                     }
                  }
               }
            }

            //Add excluded product prices to customer price after coupon application
            $totalCustomerAmountAfterCouponAndProductExclusion = $totalCustomerAmountAfterCoupon + $totalCustomerProductAmountToExclude;

            //Add excluded product prices to supplier price after coupon application
            $totalSupplierAmountAfterCouponAndProductExclusion = $totalSupplierAmountAfterCoupon + $totalSupplierProductAmountToExclude;

            $order = Order::store([
               "total_customer_amount" => $totalCustomerAmount,
               "total_customer_amount_after_coupon" => $totalCustomerAmountAfterCouponAndProductExclusion,
               "total_supplier_amount" => $totalCartAmount,
               "total_supplier_amount_after_coupon" => $totalSupplierAmountAfterCouponAndProductExclusion,
               "customer_user_id" => $user->{"id"},
               "status" => OrderStatusEnum::pending->value,
               "customer_currency_id" => $customerCurrency,
               "supplier_business_id" => $listItem[0]->{"business_id"},
               "supplier_currency_id" => $listItem[0]->{"price_currency_id"}
            ]);

            //Update order and add coupon id
            if (isset($data["coupon_code"]) && $data["coupon_code"] != null) {
               $order->updateService([
                  "coupon_id" => $coupon->{"id"},
               ]);
            }

            // Prepare elements for Order details
            foreach ($listItem as $item) {
               // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
               /** @var CartItem $cartItem */
               $cartItem = CartItem::findById($item->{"cart_item_id"});

               // Get total price for each cart item
               $totalCartItemPrice = $cartItem->computeCartItemTotalPrice();

               $customerAmount = self::handlePriceCurrencyConversion($userWallet, $totalCartItemPrice, $listItem[0]->{"price_currency_id"}, $customerCurrency);

               $orderDetail = OrderDetail::store([
                  "quantity" => $item->{"product_quantity"},
                  "customer_amount" => $customerAmount,
                  "supplier_amount" => $totalCartItemPrice,
                  "order_id" => $order->{"id"},
                  "product_id" => $item->{"product_id"},
                  "customer_currency_id" => $customerCurrency,
                  "supplier_currency_id" => $item->{"price_currency_id"},
               ]);

               // Check if card item has attributes
               $hasAttributes = $cartItem->{"cartAttributes"};

               //If Card item has attributes, create order details attributes and delete the attributes
               if ($hasAttributes != null) {
                  foreach ($hasAttributes as $attribute) {
                     $cartProductAttribute = $attribute->{"cartProductAttribute"};

                     //Create order detail attributes
                     OrderDetailAttribute::store([
                        "value" => $cartProductAttribute->{"value"},
                        "extra_value" => $cartProductAttribute->{"extra_value"},
                        "additional_price" => $cartProductAttribute->{"additional_price"},
                        "additional_price_currency_id" => $cartProductAttribute->{"additional_price_currency_id"},
                        "attribute_id" => $cartProductAttribute->{"attribute_id"},
                        "attribute_value_id" => $cartProductAttribute->{"attribute_value_id"},
                        "order_detail_id" => $orderDetail->{"id"},
                     ]);

                     //Delete cart item attributes
                     $attribute->delete();
                  }
               }

               // Delete cart item
               $cartItem->delete();
            }
         }

         DB::commit();
         $result = $order;
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::addOrder');
         DB::rollBack();

         throw  $exception;
      }

      return $result;
   }

   /**
    * Get order history
    *
    * @param String $requestType
    * @param int|null $pageNumber
    * @return LengthAwarePaginator
    */
   public static function orderList(string $requestType, int $pageNumber = null): LengthAwarePaginator
   {
      /** @var User $user */
      $user = auth()->user();
      $business = $user->{'business'};

      $queryBuilder = Order::query();

      if ($business != null) {
         $queryBuilder
            ->where(function (Builder $builder) use ($user, $business) {
               $builder
                  ->where("customer_user_id", $user->{"id"})
                  ->orWhere("supplier_business_id", $business->{"id"});
            });
      } else {
         $queryBuilder->where("customer_user_id", $user->{"id"});
      }

      if ($requestType !== OrderStatusEnum::all->value) {
         if ($requestType === OrderStatusEnum::incoming->value) {
            if ($business != null) {
               $queryBuilder
                  ->where("supplier_business_id", $business->{"id"})
                  ->where("status", OrderStatusEnum::pending->value);
            }
         } else if ($requestType === OrderStatusEnum::outgoing->value) {
            $queryBuilder
               ->where("customer_user_id", $user->{"id"})
               ->where("status", OrderStatusEnum::pending->value);
         } else {
            $queryBuilder->where("status", $requestType);
         }
      }

      return
         $queryBuilder
            ->orderByDesc('created_at')
            ->paginate(
               perPage: config("torryme.constants.items_per_page"),
               page: $pageNumber
            );
   }

   /**
    * Compute all order details
    *
    * @return array
    */
   public function computeOrderDetails(): array
   {
      $order = $this;

      return array(
         "id" => $order->{"id"},
         "type" => $order->{"type"},
         "status" => $order->{"status"},
         "order_type" => $order->{"order_type"},
         "total_customer_amount" => $order->{"total_customer_amount"},
         "total_supplier_amount" => $order->{"total_supplier_amount"},
         "customer_shipping_amount" => $order->{"customer_shipping_amount"},
         "supplier_shipping_amount" => $order->{"supplier_shipping_amount"},
         "reason_for_cancellation" => $order->{"reason_for_cancellation"},
         "created_at" => Carbon::parse($order->{'created_at'})->utc(),
         "delivery_date" => $order->{'delivery_date'} !== null ? Carbon::parse($order->{'delivery_date'})->utc() : null,
         "cancelled_at" => $order->{'cancelled_at'} !== null ? Carbon::parse($order->{'cancelled_at'})->utc() : null,
         "received_at" => $order->{'received_at'} !== null ? Carbon::parse($order->{'received_at'})->utc() : null,
         "delivered_at" => $order->{'delivered_at'} !== null ? Carbon::parse($order->{'delivered_at'})->utc() : null,

         "customer_user_currency" => $order->{"customerCurrency"}?->{'code'},
         "supplier_business_currency" => $order->{"supplierCurrency"}?->{'code'},
         "customer_shipping_currency" => $order->{"customerShippingCurrency"}?->{'code'},
         "supplier_shipping_currency" => $order->{"supplierShippingCurrency"}?->{'code'},
         "customer_user" => $order->{"customerUser"}->load('userDetail'),
         "supplier_business" => $order->{"supplierBusiness"},
         "order_details" => OrderDetail::computeOrderDetails($order),
      );
   }

   /**
    * Compute all order from paginator
    *
    * @param LengthAwarePaginator $orders
    * @return array
    */
   public static function computeOrderDetailFromPaginator(LengthAwarePaginator $orders): array
   {
      $ordersArray = array();
      foreach ($orders as $order) {
         $ordersArray[] = $order->computeOrderDetails();
      }

      return default_paginator_format(
         $orders->lastPage(),
         $orders->total(),
         $orders->currentPage(),
         "orders",
         $ordersArray,
      );
   }

   /**
    * Validate pending order
    *
    * @param array $data
    * @return Order
    * @throws InvalidOrderDeliveryDate
    * @throws InvalidOrderValidationAttempt
    */
   public static function validateOrder(array $data): Order
   {
      DB::beginTransaction();

      try {
         //Get auth user
         /** @var User $user */
         $user = auth()->user();
         $business = $user->{'business'};

         if ($business == null) {
            throw new InvalidOrderValidationAttempt(__("errors.invalid_order_validation_attempt"));
         }

         //Get order
         /** @var Order $order */
         $order = self::findById($data["order_id"]);

         //Check order validation is done by the supplier
         $order->isSupplierBusiness($business->{"id"});

         //Check delivery date is valid
         self::isValidDeliveryDate($data["delivery_date"]);

         //Get business wallet
         /** @var Wallet $businessWallet */
         $businessWallet = $user->{"wallet"};

         $customerShippingAmount = self::handlePriceCurrencyConversion($businessWallet, $data["supplier_shipping_amount"], $order->{"supplier_currency_id"}, $order->{"customer_currency_id"});

         //Update order
         $order->updateService([
            "status" => OrderStatusEnum::active->value,
            "delivery_date" => convert_datetime_to_utc($data["delivery_date"], $data[config('torryme.constants.time_zone_key')]),
            "customer_shipping_amount" => $customerShippingAmount,
            "supplier_shipping_amount" => $data["supplier_shipping_amount"],
            "customer_shipping_currency_id" => $order->{"customer_currency_id"},
            "supplier_shipping_currency_id" => $order->{"supplier_currency_id"},
         ]);

         DB::commit();
         $result = $order->refresh();
      } catch (Exception$exception) {
         log_debug(exception: $exception, prefix: 'OrderService::validateOrder');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Pay for order
    *
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws InsufficientWalletBalanceException
    * @throws Exception
    */
   public static function orderPayment(array $data): Model|Builder
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get order
      /** @var Order $order */
      $order = self::findById($data["order_id"]);

      //Check payer is owner of order
      $order->checkPayerIsOrderOwner($user->{"id"});

      //Check order has not been cancelled
      $order->hasBeenCancelled();

      //Check order has shipping cost
      if ($order->hasShippingCost()) {
         $totalDebitAmount = $order->{"total_customer_amount_after_coupon"} + $order->{"customer_shipping_amount"};
         $totalCreditAmount = $order->{"total_supplier_amount_after_coupon"} + $order->{"supplier_shipping_amount"};
      } else {
         $totalDebitAmount = $order->{"total_customer_amount_after_coupon"};
         $totalCreditAmount = $order->{"total_supplier_amount_after_coupon"};
      }

      //Check wallet has sufficient balance for payment to proceed
      $sourceWallet->inSufficientBalance($totalDebitAmount);

      //Get destination wallet
      /** @var Wallet $destinationWallet */
      $destinationWallet = $order->{"supplierBusiness"}->{"ownerUser"}->{"wallet"};

      DB::beginTransaction();
      try {
         //Get source  balance before debit
         $sourceOldBalance = $sourceWallet->getWalletBalance();

         //Debit source and get new source balance
         $sourceNewBalance = $sourceWallet->debitWallet($totalDebitAmount);

         //Update order status to request shipping
         $order->updateService(["status" => OrderStatusEnum::requestShipping->value]);

         //Put money in escrow
         $escrow = Escrow::store([
            "status" => EscrowStatusEnum::pending->value,
            "customer_amount" => $totalDebitAmount,
            "supplier_amount" => $totalCreditAmount,
            "customer_currency_id" => $order->{"customer_currency_id"},
            "supplier_currency_id" => $order->{"supplier_currency_id"},
            "source_wallet_id" => $sourceWallet->{"id"},
            "destination_wallet_id" => $destinationWallet->{"id"},
            "order_id" => $order->{"id"},
         ]);

         //Transaction type
         $transactionType = TransactionTypesNum::payment->value;
         $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

         //Get transaction exchange rate
         $exchangeRate = self::currentExchangeRate($order->{"customer_currency_id"}, $order->{"supplier_currency_id"});

         //Create transaction
         Transaction::store([
            "type" => $transactionType,
            "status" => TransactionStatusEnum::pending->value,
            "source_amount" => $totalDebitAmount,
            "source_old_balance" => $sourceOldBalance,
            "source_new_balance" => $sourceNewBalance,
            "destination_amount" => $totalCreditAmount,
            "destination_old_balance" => $destinationWallet->getWalletBalance(),
            "destination_new_balance" => null,
            "exchange_rate" => $exchangeRate,
            "transaction_reference" => transaction_reference($transactionTypeInitial),
            "payment_gateway" => PaymentGateWayEnum::internal->value,
            "source_wallet_id" => $sourceWallet->{"id"},
            "source_currency_id" => $sourceWallet->{"currency_id"},
            "destination_wallet_id" => $destinationWallet->{"id"},
            "destination_currency_id" => $destinationWallet->{"currency_id"},
            "escrow_id" => $escrow->{"id"},
         ]);

         //TODO Add platform charges
         DB::commit();
         $result = $order->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::orderPayment');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Confirm order delivery
    *
    * @param int $orderId
    * @return Order
    * @throws InvalidOrderDeliveryConfirmationAttempt
    */
   public static function confirmDelivery(int $orderId): Order
   {
      DB::beginTransaction();

      try {
         //TODO check command has not been cancelled
         //Get auth user
         /** @var User $user */
         $user = auth()->user();
         $business = $user->{'business'};

         //Get order
         /** @var Order $order */
         $order = self::findById($orderId);

         //Check command has been paid
         $order->hasBeenPaidSoCanConfirm();

         // Check request user can confirm order
         $order->canUserConfirmOrder();

         if ($business != null && $business->{"id"} === $order->{"supplier_business_id"}) {
            //Mark orders as haven been delivered
            $order->updateService(["delivered_at" => now()]);

         } elseif ($user->{"id"} === $order->{"customer_user_id"}) {
            //Check supplier has marked the order as haven been delivered
            $order->hasSupplierConfirmedDelivery();

            //Update order acknowledging delivery
            $order->updateService(["received_at" => now()]);

            //Get destination wallet
            /** @var Wallet $destinationWallet */
            $destinationWallet = $order->{"supplierBusiness"}->{"ownerUser"}->{"wallet"};

            $totalCreditAmount = $order->{"escrow"}->{"supplier_amount"};

            //Credit destination wallet and get new destination wallet balance
            $destinationNewBalance = $destinationWallet->creditWallet($totalCreditAmount);

            //Update order status to complete
            $order->updateService(["status" => OrderStatusEnum::completed->value]);

            //Update escrow status to complete
            $order->{"escrow"}->updateService(["status" => EscrowStatusEnum::completed->value]);

            //Update transaction status to complete and add transaction details for business
            $order->{"escrow"}->{"transaction"}->updateService([
               "status" => TransactionStatusEnum::completed->value,
               "destination_new_balance" => $destinationNewBalance
            ]);

         } else {
            throw new InvalidOrderDeliveryConfirmationAttempt(__("errors.invalid_order_delivery_confirmation_attempt"));
         }

         DB::commit();
         $result = $order->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::confirmDelivery');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Cancel order
    *
    * @throws InvalidOrderCancellationAttempt
    */
   public static function cancelOrder(array $data): Order
   {
      DB::beginTransaction();

      try {
         //Get auth user
         /** @var User $user */
         $user = auth()->user();
         $business = $user->{'business'};

         //Get order
         /** @var Order $order */
         $order = self::findById($data["order_id"]);

         // Check request user can cancel order
         $order->canUserCancelOrder();

         if ($order->{"status"} == OrderStatusEnum::pending->value || $order->{"status"} == OrderStatusEnum::active->value) {
            //Update order status to cancelled
            $order->updateService([
               "status" => OrderStatusEnum::cancelled->value,
               "cancelled_at" => now(),
               "reason_for_cancellation" => $data["reason_for_cancellation"],
               "cancelled_by_user_id" => $user->{"id"},
            ]);
         } else {
            if ($business != null && $business->{"id"} === $order->{"supplier_business_id"}) {
               //Check if command is eligible for cancellation by business
               $order->eligibleForBusinessCancellation();

               if ($order->{"status"} == OrderStatusEnum::requestShipping->value) {
                  //Refund client from escrow
                  $order->reversePayment($user, $data);
               }
            } else {
               //Check command has been paid
               $order->hasBeenPaidSoCanCancel();

               //Refund client from escrow
               $order->reversePayment($user, $data);
            }
         }

         DB::commit();
         $result = $order->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::cancelOrder');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Reverse payment from Escrow
    *
    * @param User $user
    * @param array $data
    */
   public function reversePayment(User $user, array $data): void
   {
      $order = $this;

      //Update order status to cancelled
      $order->updateService([
         "status" => OrderStatusEnum::cancelled->value,
         "cancelled_at" => now(),
         "reason_for_cancellation" => $data["reason_for_cancellation"],
         "cancelled_by_user_id" => $user->{"id"},
      ]);

      //Update escrow status to cancelled
      $order->{"escrow"}->updateService(["status" => EscrowStatusEnum::cancelled->value]);

      //Update transaction status to cancelled
      $order->{"escrow"}->{"transaction"}->updateService(["status" => TransactionStatusEnum::cancelled->value]);

      //Get source wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $order->{"customerUser"}->{"wallet"};

      //Get source balance before refund
      $sourceOldBalance = $sourceWallet->getWalletBalance();

      $creditAmount = $order->{"escrow"}->{"customer_amount"};

      //Credit source wallet and get new wallet balance
      $sourceNewBalance = $sourceWallet->creditWallet($creditAmount);

      //Transaction type
      $transactionType = TransactionTypesNum::refund->value;
      $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

      //Get transaction exchange rate
      $exchangeRate = self::currentExchangeRate($order->{"customer_currency_id"}, $order->{"supplier_currency_id"});

      //Create refund transaction
      Transaction::store([
         "type" => $transactionType,
         "status" => TransactionStatusEnum::completed->value,
         "source_amount" => $creditAmount,
         "source_old_balance" => $sourceOldBalance,
         "source_new_balance" => $sourceNewBalance,
         "destination_amount" => null,
         "destination_old_balance" => null,
         "destination_new_balance" => null,
         "exchange_rate" => $exchangeRate,
         "transaction_reference" => transaction_reference($transactionTypeInitial),
         "payment_gateway" => PaymentGateWayEnum::internal->value,
         "source_wallet_id" => $sourceWallet->{"id"},
         "source_currency_id" => $sourceWallet->{"currency_id"},
         "destination_wallet_id" => null,
         "destination_currency_id" => null,
         "escrow_id" => null,
      ]);
   }

   /**
    * Buy movie
    *
    * @param array $data
    * @return Order
    * @throws CouponExpiredException
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws InvalidBusinessCoupon
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buyMovie(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get movie
      $movie = Movie::findById($data["movie_id"]);

      //Get movie product
      $product = $movie->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {

         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buyMovie');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Buy series
    *
    * @param array $data
    * @return Order
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buySeries(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get series
      $series = Series::findById($data["series_id"]);

      //Get series product
      $product = $series->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {
         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buySeries');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Buy music
    *
    * @param array $data
    * @return Order
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buyMusic(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get music
      /** @var Music $music */
      $music = Music::findById($data["music_id"]);

      //Get music product
      $product = $music->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {
         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buyMusic');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Buy music album
    *
    * @param array $data
    * @return Order
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buyMusicAlbum(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get music
      /** @var MusicAlbum $musicAlbum */
      $musicAlbum = MusicAlbum::findById($data["music_album_id"]);

      //Get music album product
      $product = $musicAlbum->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {
         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buyMusicAlbum');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Buy soft copy book
    *
    * @param array $data
    * @return Order
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws InvalidBookPurchaseException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buyBook(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get book
      /** @var Book $book */
      $book = Book::findById($data["book_id"]);

      //Get book product
      $product = $book->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      $bookType = $product->{"businessSubCategoryChild"}->{"code"};

      //Check if book is hard copy
      $book->isHardCpy($bookType);

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {
         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buyBok');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Buy game
    *
    * @param array $data
    * @return Order
    * @throws DoublePurchaseException
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public static function buyGame(array $data): Order
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      //Get user wallet
      /** @var Wallet $sourceWallet */
      $sourceWallet = $user->{"wallet"};

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      //Get music
      /** @var Game $game */
      $game = Game::findById($data["game_id"]);

      //Get game product
      $product = $game->{"product"};

      //TODO Check if attributes exist and add to original price
      //Get product price
      $totalSupplierAmount = $product->{"price"};

      //Get product currency
      $productCurrency = $product->{"price_currency_id"};

      //Get customer currency
      $customerCurrency = $sourceWallet->{"currency_id"};

      //Check product has not already been purchased
      if ($product->checkProductAlreadyPurchased()) {
         throw new DoublePurchaseException(message: __('errors.double_product_purchase_attempt'));
      }

      $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

      DB::beginTransaction();
      try {
         $result = self::getResult($sourceWallet, $totalCustomerAmount, $totalSupplierAmount, $product, $user, $customerCurrency, $productCurrency, $data);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::buyGame');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Handle price conversion
    *
    * @param Wallet $sourceWallet
    * @param float $originalPrice
    * @param int $sourceCurrency
    * @param int $destinationCurrency
    * @return float|int
    */
   public static function handlePriceCurrencyConversion(Wallet $sourceWallet, float $originalPrice, int $sourceCurrency, int $destinationCurrency): float|int
   {
      //Check if source currency and destination currency are the same
      if ($sourceWallet->isSameCurrency($sourceCurrency, $destinationCurrency)) {
         $debitAmount = $originalPrice;
      } else {
         // Get exchange rate for source and destination currencies
         $exchangeRate = self::currentExchangeRate($sourceCurrency, $destinationCurrency);

         // Multiply original price by exchange rate to get corresponding amount in destination currency
         $debitAmount = $originalPrice * $exchangeRate;
      }

      return $debitAmount;
   }

   /**
    * Handle coupon currency conversion
    *
    * @param $couponCurrency
    * @param $couponAmount
    * @param $destinationCurrency
    * @return mixed
    */
   public static function handleCouponCurrencyConversion($couponCurrency, $couponAmount, $destinationCurrency): mixed
   {
      if ($couponCurrency === $destinationCurrency) {
         $amountAfterCouponApplication = $couponAmount;
      } else {
         // Get exchange rate for coupon and destination currencies
         $exchangeRate = self::currentExchangeRate($couponCurrency, $destinationCurrency);

         // Multiply coupon amount by exchange rate to get corresponding amount in destination currency
         $amountAfterCouponApplication = $couponAmount * $exchangeRate;
      }

      return $amountAfterCouponApplication;
   }

   /**
    * Handle coupon mina dn max spending amount conversion
    *
    * @param Coupon $coupon
    * @param int $totalCustomerAmount
    * @param int $customerCurrency
    * @throws InvalidCouponAmountToSpendException
    */
   public static function handleCouponMinAndMaxAmountConversion(Coupon $coupon, int $totalCustomerAmount, int $customerCurrency)
   {
      if ($coupon->{"minimum_amount_to_spend"} != null && $coupon->{"maximum_amount_to_spend"} != null) {
         //Convert coupon minimum amount to spend if coupon currency and customer currency are different
         $couponMinimumAmountToSpend = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $coupon->{"minimum_amount_to_spend"}, $customerCurrency);

         if ($totalCustomerAmount < $couponMinimumAmountToSpend) {
            throw new InvalidCouponAmountToSpendException(__("errors.cart_amount_less_than_coupon_minimum_amount"));
         }

         //Convert coupon maximum amount to spend if coupon currency and customer currency are different
         $couponMaximumAmountToSpend = self::handleCouponCurrencyConversion($coupon->{"maximum_amount_to_spend_currency_id"}, $coupon->{"maximum_amount_to_spend"}, $customerCurrency);

         if ($totalCustomerAmount > $couponMaximumAmountToSpend) {
            throw new InvalidCouponAmountToSpendException(__("errors.cart_amount_greater_than_coupon_maximum_amount"));
         }
      }
   }

   /**
    * Handle coupon product exclusion
    *
    * @param $listItem
    * @param Coupon $coupon
    * @param Wallet $sourceWallet
    * @param int $sourceCurrency
    * @param int $destinationCurrency
    * @param int $userId
    * @return float|int|void
    */
   public static function handleCustomerProductExclusion($listItem, Coupon $coupon, Wallet $sourceWallet, int $sourceCurrency, int $destinationCurrency, int $userId)
   {
      if (sizeof($coupon->{"productExclusion"}) != 0) {
         $totalProductAmountToExcluded = array();
         foreach ($listItem as $item) {
            // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
            /** @var CartItem $cartItem */
            $cartItem = CartItem::findById($item->{"cart_item_id"});
            foreach ($coupon->{"productExclusion"} as $productExclusion) {
               //Check if cart item is the excluded product
               if ($cartItem->{"product_id"} == $productExclusion->{"product"}->{"id"}) {

                  // Get total price for each product to be excluded
                  $totalCartItemPriceToExclude = $cartItem->computeCartItemTotalPrice();

                  //Join all prices to be excluded
                  $totalProductAmountToExcluded[] = $totalCartItemPriceToExclude;
               }
            }
         }

         $totalCartAmountToExclude = array_sum($totalProductAmountToExcluded);

         return self::handlePriceCurrencyConversion($sourceWallet, $totalCartAmountToExclude, $sourceCurrency, $destinationCurrency);
      }
   }

   /**
    * Handle supplier product exclusion
    *
    * @param $listItem
    * @param Coupon $coupon
    * @return float|int|void
    */
   public static function handleSupplierProductExclusion($listItem, Coupon $coupon)
   {
      if (sizeof($coupon->{"productExclusion"}) != 0) {
         $totalSupplierProductAmountToExcluded = array();
         foreach ($listItem as $item) {
            // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
            /** @var CartItem $cartItem */
            $cartItem = CartItem::findById($item->{"cart_item_id"});

            foreach ($coupon->{"productExclusion"} as $productExclusion) {
               //Get product associated with exclusion
               $associatedProduct = $productExclusion->{"product"};

               //Check if cart item is the excluded product
               if ($cartItem->{"product_id"} == $associatedProduct->{"id"}) {

                  // Get total price for each product to be excluded
                  $totalCartItemPriceToExclude = $cartItem->computeCartItemTotalPrice();

                  //Join all prices to be excluded
                  $totalSupplierProductAmountToExcluded[] = $totalCartItemPriceToExclude;
               }
            }
         }

         return array_sum($totalSupplierProductAmountToExcluded);
      }
   }

   /**
    * Get exchange rate
    *
    * @param int $sourceCurrency
    * @param int $destinationCurrency
    * @return mixed
    */
   public static function currentExchangeRate(int $sourceCurrency, int $destinationCurrency): mixed
   {
      $exchangeRate = ExchangeRate::getExchangeRate($sourceCurrency, $destinationCurrency);

      return $exchangeRate->{"exchange_rate"};
   }

   /**
    * Handle purchase results
    *
    * @param Wallet $sourceWallet
    * @param float|int $totalCustomerAmount
    * @param float|int $totalSupplierAmount
    * @param mixed $product
    * @param User $user
    * @param int $customerCurrency
    * @param int $productCurrency
    * @param array $data
    * @return Order
    * @throws CouponExpiredException
    * @throws InsufficientWalletBalanceException
    * @throws InvalidBusinessCoupon
    * @throws InvalidCouponAmountToSpendException
    */
   public static function getResult(Wallet $sourceWallet, float|int $totalCustomerAmount, float|int $totalSupplierAmount, mixed $product, User $user, int $customerCurrency, int $productCurrency, array $data): Order
   {
      //In cases where there are no discounts, amount before and amount after discount should be the same
      $totalCustomerAmountAfterCoupon = $totalCustomerAmount;
      $totalSupplierAmountAfterCoupon = $totalSupplierAmount;

      //Initiate variables
      $totalCustomerProductAmountToExclude = 0;
      $totalSupplierProductAmountToExclude = 0;

      if (isset($data["coupon_code"]) && $data["coupon_code"] != null) {

         /** @var Coupon $coupon */
         $coupon = Coupon::findByCode($data["coupon_code"]);

         //Check if coupon has expired
         $coupon->hasExpired();

         //Check if coupon belong to business product
         $coupon->isValidBusinessCoupon($product->{"business_id"});

         // Apply percentage card discount to total order amount
         if ($coupon->isPercentageCartDiscount()) {

            self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmountAfterCoupon, $customerCurrency);

            //Get customer percentage discount amount of customer amount after product exclusion
            $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmount;

            //Deduct coupon amount from customer amount after product exclusion
            $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerPercentageCardDiscountAmount;

            //Get supplier percentage discount amount of cart amount
            $suppliedPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalSupplierAmount;

            //Deduct coupon amount from supplier amount after product exclusion
            $totalSupplierAmountAfterCoupon = $totalSupplierAmount - $suppliedPercentageCardDiscountAmount;
         }

         // Apply fixed card discount to total card amount
         if ($coupon->isFixedCartDiscount()) {

            self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);

            //Convert coupon amount to customer currency
            $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $coupon->{"fixed_card_discount_amount"}, $customerCurrency);
            //Deduct coupon amount from customer amount after product exclusion
            $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerCouponAmount;

            //Convert coupon amount to supplier currency
            $supplierCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $coupon->{"fixed_card_discount_amount"}, $product->{"price_currency_id"});
            //Deduct coupon amount from supplier amount after product exclusion
            $totalSupplierAmountAfterCoupon = $totalSupplierAmount - $supplierCouponAmount;
         }

         //Apply percentage discount to product
         if ($coupon->isProductDiscountByPercentage()) {
            if ($coupon->{"product_id"} == $product->{"id"}) {
               self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);

               //Get customer percentage discount amount of total command
               $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmount;

               //Deduct coupon amount from customer amount
               $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerPercentageCardDiscountAmount;

               //Get supplier percentage discount amount of total command
               $suppliedPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalSupplierAmount;

               //Deduct coupon amount from supplier amount
               $totalSupplierAmountAfterCoupon = $totalSupplierAmount - $suppliedPercentageCardDiscountAmount;
            }
         }

         // Apply fix amount discount to product
         if ($coupon->isProductDiscountByFixedAmount()) {
            if ($coupon->{"product_id"} == $product->{"id"}) {
               self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);
               //Get product discount amount
               $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

               //Deduct coupon amount from supplier amount
               $totalSupplierAmountAfterCoupon = $totalSupplierAmount - $productDiscountAmount;

               //Convert coupon amount to customer currency
               $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);

               //Deduct coupon amount from customer amount
               $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerCouponAmount;
            }
         }

      }

      //Check wallet has sufficient balance for payment to proceed
      $sourceWallet->inSufficientBalance($totalCustomerAmountAfterCoupon);

      //Get source  balance before debit
      $sourceOldBalance = $sourceWallet->getWalletBalance();

      //Debit source and get new source balance
      $sourceNewBalance = $sourceWallet->debitWallet($totalCustomerAmountAfterCoupon);

      //Get destination wallet
      /** @var Wallet $destinationWallet */
      $destinationWallet = $product->{"business"}->{"ownerUser"}->{"wallet"};

      //Get destination wallet balance before credit
      $destinationOldBalance = $destinationWallet->getWalletBalance();

      //Credit destination wallet and get new destination wallet balance
      $destinationNewBalance = $destinationWallet->creditWallet($totalSupplierAmountAfterCoupon);

      /** @var Order $order */
      $order = Order::store([
         "total_customer_amount" => $totalCustomerAmount,
         "total_customer_amount_after_coupon" => $totalCustomerAmountAfterCoupon,
         "total_supplier_amount" => $totalSupplierAmount,
         "total_supplier_amount_after_coupon" => $totalSupplierAmountAfterCoupon,
         "customer_user_id" => $user->{"id"},
         "type" => OrderTypesEnum::simple->value,
         "status" => OrderStatusEnum::completed->value,
         "customer_currency_id" => $customerCurrency,
         "supplier_business_id" => $product->{"business_id"},
         "supplier_currency_id" => $productCurrency
      ]);

      OrderDetail::store([
         "quantity" => $product->{"quantity"},
         "customer_amount" => $totalCustomerAmount,
         "supplier_amount" => $totalSupplierAmount,
         "order_id" => $order->{"id"},
         "product_id" => $product->{"id"},
         "customer_currency_id" => $customerCurrency,
         "supplier_currency_id" => $productCurrency,
      ]);

      //Transaction type
      $transactionType = TransactionTypesNum::payment->value;
      $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

      //Get transaction exchange rate
      $exchangeRate = self::currentExchangeRate($productCurrency, $customerCurrency);

      //Create transaction
      Transaction::store([
         "type" => $transactionType,
         "status" => TransactionStatusEnum::completed->value,
         "source_amount" => $totalCustomerAmountAfterCoupon,
         "source_old_balance" => $sourceOldBalance,
         "source_new_balance" => $sourceNewBalance,
         "destination_amount" => $totalSupplierAmountAfterCoupon,
         "destination_old_balance" => $destinationOldBalance,
         "destination_new_balance" => $destinationNewBalance,
         "exchange_rate" => $exchangeRate,
         "transaction_reference" => transaction_reference($transactionTypeInitial),
         "payment_gateway" => PaymentGateWayEnum::internal->value,
         "source_wallet_id" => $sourceWallet->{"id"},
         "source_currency_id" => $customerCurrency,
         "destination_wallet_id" => $destinationWallet->{"id"},
         "destination_currency_id" => $destinationWallet->{"currency_id"},
         "transaction_parent_id" => null,
         "escrow_id" => null,
      ]);

      //Delete cart item if it exists
      $cartItem = CartItem::getUserCartItem($user, $product);
      $cartItem?->delete();

      DB::commit();
      return $order->refresh();
   }

   /**
    * Get auth user purchases
    *
    * @param string|null $category
    * @param $pageNumber
    * @return LengthAwarePaginator
    */
   public static function purchases(string $category = null, $pageNumber = null): LengthAwarePaginator
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();

      $queryBuilder =
         Order::query()
            ->where("customer_user_id", $user->{"id"})
            ->where("status", OrderStatusEnum::completed->value);

      if ($category != null) {
         $queryBuilder = $queryBuilder->whereHas("orderDetails.product.businessCategory", function (Builder $query) use ($category) {
            $query->where('code', $category);
         });
      }

      return
         $queryBuilder
            ->orderByDesc('created_at')
            ->paginate(
               perPage: config("torryme.constants.items_per_page"),
               page: $pageNumber
            );
   }

   /**
    * Get business sales
    *
    * @param string|null $category
    * @param int|null $pageNumber
    * @return LengthAwarePaginator
    */
   public static function sales(string $category = null, int $pageNumber = null): LengthAwarePaginator
   {
      //Get auth user
      /** @var User $user */
      $user = auth()->user();
      $business = $user->{'business'};

      $queryBuilder =
         Order::query()
            ->where("supplier_business_id", $business->{"id"})
            ->where("status", OrderStatusEnum::completed->value);

      if ($category != null) {
         $queryBuilder = $queryBuilder->whereHas("orderDetails.product.businessCategory", function (Builder $query) use ($category) {
            $query->where('code', $category);
         });
      }

      return
         $queryBuilder
            ->orderByDesc('created_at')
            ->paginate(
               perPage: config("torryme.constants.items_per_page"),
               page: $pageNumber
            );
   }

   /**
    * Check if user has purchased product with given id
    *
    * @param int $userId
    * @param int $productId
    * @return Model|Builder|null
    */
   public static function checkProductPurchase(int $userId, int $productId): Model|Builder|null
   {
      return
         Order::query()
            ->where("customer_user_id", $userId)
            ->where("status", OrderStatusEnum::completed->value)
            ->whereHas("orderDetails", function (Builder $query) use ($productId) {
               $query->where('product_id', $productId);
            })
            ->first();
   }

   /**
    * Compute coupon amount
    *
    * @param array $data
    * @return array|null
    * @throws CouponExpiredException
    * @throws InvalidBusinessCoupon
    * @throws InvalidCouponAmountToSpendException
    */
   public static function computeCouponAmount(array $data): ?array
   {
      try {
         //Get auth user
         /** @var User $user */
         $user = auth()->user();

         //Get user wallet
         /** @var Wallet $sourceWallet */
         $sourceWallet = $user->{"wallet"};

         $customerCurrency = $sourceWallet->{"currency_id"};

         /** @var Coupon $coupon */
         $coupon = Coupon::findByCode($data["coupon_code"]);

         $totalCustomerAmount = 0;
         $totalCustomerAmountAfterCoupon = 0;

         if (isset($data["cart_ids"]) && $data["cart_ids"] != null) {

            // Get all cart items and their corresponding products, grouped by business id
            $cartItemList = CartItem::getCartItemListByBusiness($data["cart_ids"]);

            foreach ($cartItemList as $listItem) {
               $totalCartPrice = array();
               foreach ($listItem as $item) {
                  // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
                  /** @var CartItem $cartItem */
                  $cartItem = CartItem::findById($item->{"cart_item_id"});

                  // Get total price for each cart item
                  $totalCartItemPrice = $cartItem->computeCartItemTotalPrice();

                  // Join all cart item prices for a business together
                  $totalCartPrice[] = $totalCartItemPrice;
               }

               // Sum all cart item prices for a business
               $totalCartAmount = array_sum($totalCartPrice);

               $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalCartAmount, $listItem[0]->{"price_currency_id"}, $customerCurrency);

               //In cases where there are no discounts, amount before and amount after discount should be the same
               $totalCustomerAmountAfterCoupon = $totalCustomerAmount;

               //Check if coupon has expired
               $coupon->hasExpired();

               //Check if coupon belong to business product
               $coupon->isValidBusinessCoupon($listItem[0]->{"business_id"});

               // Apply percentage card discount to total order amount
               if ($coupon->isPercentageCartDiscount()) {

                  self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmountAfterCoupon, $customerCurrency);

                  //Handle customer product exclusion
                  $totalCustomerProductAmountToExclude = self::handleCustomerProductExclusion($listItem, $coupon, $sourceWallet, $listItem[0]->{"price_currency_id"}, $customerCurrency, $user->{"id"});

                  //Subtract total product exclusion amount from customer original cart amount
                  $totalCustomerAmountAfterProductExclusion = $totalCustomerAmount - $totalCustomerProductAmountToExclude;

                  //Get customer percentage discount amount of customer amount after product exclusion
                  $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmountAfterProductExclusion;

                  //Deduct coupon amount from customer amount after product exclusion
                  $totalCustomerAmountAfterCoupon = $totalCustomerAmountAfterProductExclusion - $customerPercentageCardDiscountAmount;

               }

               // Apply fixed card discount to total order amount
               if ($coupon->isFixedCartDiscount()) {

                  self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmountAfterCoupon, $customerCurrency);

                  //Handle product exclusion
                  $totalCustomerProductAmountToExclude = self::handleCustomerProductExclusion($listItem, $coupon, $sourceWallet, $listItem[0]->{"price_currency_id"}, $customerCurrency, $user->{"id"});

                  //Subtract total product exclusion amount from original cart amount
                  $totalCustomerAmountAfterProductExclusion = $totalCustomerAmount - $totalCustomerProductAmountToExclude;

                  //Get product discount amount
                  $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

                  //Convert coupon amount to customer currency
                  $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);
                  //Deduct coupon amount from customer amount after product exclusion
                  $totalCustomerAmountAfterCoupon = $totalCustomerAmountAfterProductExclusion - $customerCouponAmount;
               }

               //Apply percentage discount to product
               if ($coupon->isProductDiscountByPercentage()) {
                  //Initiate variables
                  $customerPercentageCardDiscountAmount = 0;
                  $customerProductAmountWithoutCoupon = 0;

                  foreach ($listItem as $item) {
                     // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
                     /** @var CartItem $cartItem */
                     $cartItem = CartItem::findById($item->{"cart_item_id"});

                     if ($coupon->{"product_id"} == $cartItem->{"product"}->{"id"}) {
                        //Get product price
                        $totalProductItemPrice = $cartItem->computeCartItemTotalPrice();

                        //Get product price in customer currency
                        $totalCustomerProductAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalProductItemPrice, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);

                        //Get customer percentage discount amount of total command
                        $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerProductAmount;
                     } else {
                        //Get product price without discount in supplier currency
                        $productPriceWithOutDiscount = $cartItem->computeCartItemTotalPrice();

                        //Get product price without discount in customer currency
                        $customerProductAmountWithoutCoupon = self::handlePriceCurrencyConversion($sourceWallet, $productPriceWithOutDiscount, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);
                     }
                  }

                  //Add coupon discount amount to product amount without discount
                  $totalCustomerAmountAfterCoupon = $customerPercentageCardDiscountAmount + $customerProductAmountWithoutCoupon;
               }

               // Apply fix amount discount to product
               if ($coupon->isProductDiscountByFixedAmount()) {
                  //Get product discount amount
                  $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

                  //Initiate variables
                  $customerProductAmountAfterCoupon = 0;
                  $customerProductAmountWithoutCoupon = 0;

                  foreach ($listItem as $item) {
                     // Find by cart id to get cart item. Since $item is a join of cart_items and products tables
                     /** @var CartItem $cartItem */
                     $cartItem = CartItem::findById($item->{"cart_item_id"});

                     if ($coupon->{"product_id"} == $cartItem->{"product"}->{"id"}) {
                        //Get product price in supplier currency
                        $totalProductItemPrice = $cartItem->computeCartItemTotalPrice();

                        //Get product price in customer currency
                        $customerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalProductItemPrice, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);

                        //Convert coupon amount to customer currency
                        $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);

                        //Deduct coupon amount from customer amount
                        $customerProductAmountAfterCoupon = $customerAmount - $customerCouponAmount;
                     } else {
                        //Get product price without discount in supplier currency
                        $productPriceWithOutDiscount = $cartItem->computeCartItemTotalPrice();

                        //Get product price without discount in customer currency
                        $customerProductAmountWithoutCoupon = self::handlePriceCurrencyConversion($sourceWallet, $productPriceWithOutDiscount, $cartItem->{"product"}->{"price_currency_id"}, $customerCurrency);
                     }
                  }

                  //Add coupon discount amount to product amount without discount
                  $totalCustomerAmountAfterCoupon = $customerProductAmountAfterCoupon + $customerProductAmountWithoutCoupon;
               }
            }
         }

         if (isset($data["movie_id"]) && $data["movie_id"] != null) {
            //Get movie
            $movie = Movie::findById($data["movie_id"]);

            //Get movie product
            $product = $movie->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         if (isset($data["series_id"]) && $data["series_id"] != null) {
            //Get series
            $series = Series::findById($data["series_id"]);

            //Get series product
            $product = $series->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         if (isset($data["music_id"]) && $data["music_id"] != null) {
            //Get music
            /** @var Music $music */
            $music = Music::findById($data["music_id"]);

            //Get music product
            $product = $music->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         if (isset($data["music_album_id"]) && $data["music_album_id"] != null) {
            //Get music
            /** @var MusicAlbum $musicAlbum */
            $musicAlbum = MusicAlbum::findById($data["music_album_id"]);

            //Get music album product
            $product = $musicAlbum->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         if (isset($data["book_id"]) && $data["book_id"] != null) {
            //Get book
            /** @var Book $book */
            $book = Book::findById($data["book_id"]);

            //Get book product
            $product = $book->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         if (isset($data["game_id"]) && $data["game_id"] != null) {
            //Get music
            /** @var Game $game */
            $game = Game::findById($data["game_id"]);

            //Get game product
            $product = $game->{"product"};

            //Get product currency
            $productCurrency = $product->{"price_currency_id"};

            //TODO Check if attributes exist and add to original price
            //Get product price
            $totalSupplierAmount = $product->{"price"};

            $totalCustomerAmount = self::handlePriceCurrencyConversion($sourceWallet, $totalSupplierAmount, $productCurrency, $customerCurrency);

            $totalCustomerAmountAfterCoupon = self::handleComputeCouponAmount($coupon, $product, $totalCustomerAmount, $customerCurrency);
         }

         //Calculate product discount
         $discount = $totalCustomerAmountAfterCoupon != 0 ? ($totalCustomerAmount - $totalCustomerAmountAfterCoupon) : 0;

         $finalData = array(
            "product_price" => $totalCustomerAmount,
            "price_after_discount" => $totalCustomerAmountAfterCoupon,
            "discount" => $discount,
            "coupon" => $coupon,
         );

         $result = $finalData;
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'OrderService::computeCouponReduction');
         throw $exception;
      }

      return $result;
   }

   /**
    * @param Coupon $coupon
    * @param mixed $product
    * @param float|int $totalCustomerAmount
    * @param int $customerCurrency
    * @return mixed
    * @throws CouponExpiredException
    * @throws InvalidBusinessCoupon
    * @throws InvalidCouponAmountToSpendException
    */
   public static function handleComputeCouponAmount(Coupon $coupon, Mixed $product, float|int $totalCustomerAmount, int $customerCurrency): mixed
   {

      $totalCustomerAmountAfterCoupon = $totalCustomerAmount;

      //Check if coupon has expired
      $coupon->hasExpired();

      //Check if coupon belong to business product
      $coupon->isValidBusinessCoupon($product->{"business_id"});

      // Apply percentage card discount to total order amount
      if ($coupon->isPercentageCartDiscount()) {

         self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);

         //Get customer percentage discount amount of customer amount after product exclusion
         $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmount;

         //Deduct coupon amount from customer amount after product exclusion
         $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerPercentageCardDiscountAmount;
      }

      // Apply fixed card discount to total order amount
      if ($coupon->isFixedCartDiscount()) {

         self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);

         //Get product discount amount
         $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

         //Convert coupon amount to customer currency
         $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);

         //Deduct coupon amount from customer amount after product exclusion
         $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerCouponAmount;
      }

      //Apply percentage discount to product
      if ($coupon->isProductDiscountByPercentage()) {
         if ($coupon->{"product_id"} == $product->{"id"}) {
            self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);

            //Get customer percentage discount amount of total command
            $customerPercentageCardDiscountAmount = ($coupon->{"percentage_discount"} / 100) * $totalCustomerAmount;

            //Deduct coupon amount from customer amount
            $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerPercentageCardDiscountAmount;
         }
      }

      // Apply fix amount discount to product
      if ($coupon->isProductDiscountByFixedAmount()) {
         if ($coupon->{"product_id"} == $product->{"id"}) {
            self::handleCouponMinAndMaxAmountConversion($coupon, $totalCustomerAmount, $customerCurrency);
            //Get product discount amount
            $productDiscountAmount = $coupon->{"fixed_card_discount_amount"};

            //Convert coupon amount to customer currency
            $customerCouponAmount = self::handleCouponCurrencyConversion($coupon->{"minimum_amount_to_spend_currency_id"}, $productDiscountAmount, $customerCurrency);

            //Deduct coupon amount from customer amount
            $totalCustomerAmountAfterCoupon = $totalCustomerAmount - $customerCouponAmount;
         }
      }

      return $totalCustomerAmountAfterCoupon;
   }

}
