<?php

namespace App\Services;

use App\Enums\MusicUploadKeyEnum;
use App\Exceptions\UploadFileException;
use App\Models\Business;
use App\Models\Music;
use App\Models\MusicAlbum;
use App\Models\Product;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;

trait MusicAlbumService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return MusicAlbum::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return MusicAlbum::query()->find($id);
    }

    /**
     * Add new music album
     *
     * @param UploadedFile $albumPoster
     * @param Product $product
     * @param Business $business
     * @return Builder|Model
     * @throws UploadFileException
     */
    public static function createMusicAlbum(UploadedFile $albumPoster, Product $product, Business $business): Model|Builder
    {
        $basePath = sprintf(config('torryme.paths.docs'), $business->buildProductDir($product->{'id'}));

        // Try to upload file ...
        $resultPath = upload_file_system(
            MusicUploadKeyEnum::albumPoster->value,
            $basePath,
            $albumPoster,
        );

        if(! filled($resultPath)) {
            throw new UploadFileException();
        }

        return MusicAlbum::store(array(
            'cover' => $resultPath,
            'designation' => $product->{'designation'},
            'description' => $product->{'description'},
            'price' => $product->{'price'},
            'price_currency_id' => $product->{'price_currency_id'},
        ));
    }

    /**
     * Add new music single
     *
     * @param UploadedFile $musicFile
     * @param array $data
     * @return Builder|Model|null
     * @throws \Exception
     */
    public function addMusicSingle(UploadedFile $musicFile, array $data): Model|Builder|null
    {
        return Music::createNewMusicSingleForAlbum($this, $musicFile, $data);
    }

    /**
     * Total number of music single into the album
     *
     * @return int
     */
    public function totalAlbumMusic(): int
    {
        return Music::totalAlbumMusic($this);
    }
}
