<?php

namespace App\Services;

use App\Enums\DisputeStatusEnum;
use App\Exceptions\DisputeAlreadyUnderwayException;
use App\Models\Dispute;
use App\Models\DisputeParticipant;
use App\Models\Transaction;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait DisputeService
{
   /**
    * Create new
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return Dispute::query()->create($data);
   }

   /**
    * Serialize for channel
    *
    * @return array
    */
   public function serializeForChannel(): array
   {
      $participants = [];
      $allParticipants = $this->{'participants'};
      $disputeType = $this->{'disputeType'};

      foreach ($allParticipants as $participant) {
         $participants[] = $participant->load(['user.userDetail', 'user.business']);
      }

      return array(
         'uuid' => $this->{'uuid'},
         'name' => $this->{'name'},
         'label_color' => $this->{'label_color'},
         'type' => $disputeType->{'designation'},
         'status' => $this->{'status'},
         'participants' => $participants,
         'creator' => $this->{'creator'}->load(['userDetail', 'business']),
         'transaction' => $this->{'transaction'}
      );
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Find by uuid
    *
    * @param string $uuid
    * @return mixed
    */
   public static function findByUuid(string $uuid): mixed
   {
      return Dispute::query()->where('uuid', $uuid)->first();
   }

   /**
    * Create new dispute
    *
    * @param array $data
    * @return Model|Builder|null
    * @throws DisputeAlreadyUnderwayException
    */
   public static function addNewDispute(array $data): Model|Builder|null
   {
      $result = null;
      DB::beginTransaction();

      try {
         // Find
         $dispute = self::findByTransactionAndCreator(
            auth()->id(),
            $data['transaction_id']
         );

         if($dispute !== null) {
            throw new DisputeAlreadyUnderwayException();
         }

         // Compute dispute name
         $name = $data['dispute_name'] ?? '';

         /** @var Dispute $dispute */
         $dispute = self::store([
            'uuid' => generate_uuid('disputes'),
            'name' => $name,
            'label_color' => $data['label_color'],
            'status' => DisputeStatusEnum::onGoing,

            'created_by_id' => auth()->id(),
            'transaction_id' => $data['transaction_id'],
            'dispute_type_id' => $data['dispute_type_id'],
         ]);

         $transaction = Transaction::findById($data['transaction_id']);
         // Add participants
         $participantIds = array(
            $transaction->{'destinationWallet'}->{'user_id'},
            $transaction->{'sourceWallet'}->{'user_id'},
         );
         // TODO Add admin user
         $participantIds = array_unique($participantIds);

         // Create participants
         $participants = [];
         foreach ($participantIds as $id) {
            $participants[] = DisputeParticipant::store([
               'dispute_id' => $dispute->{'id'},
               'participant_id' => $id
            ]);
         }

         if (!filled($name)) {
            $phones = [];
            foreach ($participants as $participant) {
               $phones[] = $participant->{'user'}->{'telephone'};
            }
            $dispute->updateService([
               'name' => implode(', ', $phones)
            ]);
         }

         DB::commit();

         /** @var Dispute $dispute */
         $dispute = $dispute->refresh();

         // Notify participants ...
         $dispute->sendDisputeInvitationNotification();

         $result = $dispute;
      } catch (\Exception $exception) {
         DB::rollBack();
         log_debug($exception, 'DisputeService::addNewDispute Unable to add new dispute');

         throw $exception;
      }

      return $result;
   }

   /**
    * We check if there is a transaction in which the
    *  user is already a participant
    *
    * @param int $participantUserId
    * @param int $transactionId
    * @return Builder|Model|null
    */
   public static function findByTransactionAndCreator(int $participantUserId, int $transactionId): Model|Builder|null
   {
      return
         Dispute::query()
            ->where('status', DisputeStatusEnum::onGoing->{'value'})
            ->where('transaction_id', $transactionId)
            ->whereHas('participants.user', function (Builder $builder) use($participantUserId) {
               $builder->where('id', $participantUserId);
            })
            ->first();
   }

   /**
    * Get dispute history
    *
    * @return array
    */
   public static function history(): array
   {
      $disputes = [];
      $data =
         Dispute::query()
            ->with('latestMessages')
            ->whereHas('participants.user', function (Builder $builder) {
               $builder->where('id', auth()->id());
            })
            ->orderBy('created_at', 'desc')
            ->get();

      foreach ($data as $disputeItem) {
         $messages = $disputeItem->{'latestMessages'};
         $messagesResult = [];

         foreach ($messages as $message) {
            $messagesResult[] = $message->serializeForChannel();
         }

         $disputeItem->{'latest_messages'} = $messagesResult;
         $disputes[] = [
            ...$disputeItem->serializeForChannel(),
            'latest_messages' => $messagesResult
         ];
      }

      return $disputes;
   }
}
