<?php

namespace App\Services;

use App\Models\BusinessCategory;
use App\Models\BusinessSubCategory;
use App\Models\BusinessSubCategoryChild;
use App\Models\ContentRestriction;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Mockery\Exception;

trait ContentRestrictionService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return ContentRestriction::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * FInd by ID
     *
     * @param $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById($id): Model|Collection|Builder|array|null
    {
        return ContentRestriction::query()->find($id);
    }

    /**
     * Add restrictions
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function addRestriction(array $data): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();

        $result = null;
        try {

            if (isset($data["keywords"])) {
                foreach ($data["keywords"] as $words) {
                    $yeyWord = self::keyWordExist($user->{"id"}, $words);
                    if ($yeyWord == null) {
                        $result = self::store([
                            "keyword" => $words,
                            "user_id" => $user->{"id"},
                        ]);
                    } else {
                        $result = $yeyWord;
                    }
                }
            }


            if (isset($data["categories"])) {
                foreach ($data["categories"] as $category) {

                    if (isset($category["business_category_id"])) {
                        $isValidId = BusinessCategory::findById($category["business_category_id"]);
                        if ($isValidId != null) {
                            $businessCategory = self::businessCategoryExit($user->{"id"}, $category["business_category_id"]);
                            if ($businessCategory == null) {
                                $result = self::store([
                                    "business_category_id" => $category["business_category_id"],
                                    "user_id" => $user->{"id"},
                                ]);
                            } else {
                                $result = $businessCategory;
                            }
                        }
                    }

                    if (isset($category["business_sub_category_id"])) {
                        $isValidId = BusinessSubCategory::findById($category["business_sub_category_id"]);
                        if ($isValidId != null) {
                            $businessSubCategory = self::businessSubCategoryExist($user->{"id"}, $category["business_sub_category_id"]);
                            if ($businessSubCategory == null) {
                                $result = self::store([
                                    "business_sub_category_id" => $category["business_sub_category_id"],
                                    "user_id" => $user->{"id"},
                                ]);
                            } else {
                                $result = $businessSubCategory;
                            }
                        }
                    }

                    if (isset($category["business_sub_category_child_id"])) {
                        $isValidId = BusinessSubCategoryChild::findById($category["business_sub_category_child_id"]);
                        if ($isValidId != null) {
                            $businessSubCategoryChild = self::businessSubCategoryChildExist($user->{"id"}, $category["business_sub_category_child_id"]);
                            if ($businessSubCategoryChild == null) {
                                $result = self::store([
                                    "business_sub_category_child_id" => $category["business_sub_category_child_id"],
                                    "user_id" => $user->{"id"},
                                ]);
                            } else {
                                $result = $businessSubCategoryChild;
                            }
                        }
                    }

                }
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ContentRestrictionService::add');
        }

        return $result;
    }

    /**
     * Check if keyword exist
     *
     * @param int $userId
     * @param string $keyWord
     * @return Model|Builder|null
     */
    public static function keyWordExist(int $userId, string $keyWord): Model|Builder|null
    {
        return ContentRestriction::query()->where('user_id', $userId)->where('keyword', $keyWord)->first();
    }

    /**
     * Check if business category exist
     *
     * @param int $userId
     * @param int $businessCategoryId
     * @return Model|Builder|null
     */
    public static function businessCategoryExit(int $userId, int $businessCategoryId): Model|Builder|null
    {
        return ContentRestriction::query()->where('user_id', $userId)->where('business_category_id', $businessCategoryId)->first();
    }

    /**
     * Check if business sun category exist
     *
     * @param int $userId
     * @param int $businessSubCategoryId
     * @return Model|Builder|null
     */
    public static function businessSubCategoryExist(int $userId, int $businessSubCategoryId): Model|Builder|null
    {
        return ContentRestriction::query()->where('user_id', $userId)->where('business_sub_category_id', $businessSubCategoryId)->first();
    }

    /**
     * Check if business sib category child exist
     *
     * @param int $userId
     * @param int $businessSubCategoryChildId
     * @return Model|Builder|null
     */
    public static function businessSubCategoryChildExist(int $userId, int $businessSubCategoryChildId): Model|Builder|null
    {
        return ContentRestriction::query()->where('user_id', $userId)->where('business_sub_category_child_id', $businessSubCategoryChildId)->first();
    }

    /**
     * Get content restriction
     *
     * @return Builder[]|Collection
     */
    public static function getRestrictions(): Collection|array
    {
        /** @var User $user */
        $user = auth()->user();

        return
            ContentRestriction::query()
                ->with(['businessCategory', 'businessSubCategory', 'businessSubCategoryChild'])
                ->where("user_id", $user->{"id"})
                ->orderByDesc('created_at')
                ->get();
    }

    /**
     * Compute category tree for content restrictions
     *
     * @param $category
     * @return array
     */
    public static function computeCategoryTreeForRestrictions($category): array
    {
        $items = new Collection();
        $type = '';

        if($category instanceof BusinessCategory) {
            $type = 'business_category';
            $items = $category->{'businessSubCategories'};
        }

        if($category instanceof BusinessSubCategory) {
            $type = 'business_sub_category';
            $items = $category->businessSubCategoriesChild()->whereNull('parent_id')->get();
        }

        if($category instanceof BusinessSubCategoryChild) {
            $type = 'business_sub_category_child';
            $items = $category->businessSubCategoriesChild()->whereNotNull('parent_id')->get();
        }

        $result = [
            'id' => $category->{'id'},
            'designation' => $category->{'designation'},
            'type' => $type,
            'children' => []
        ];

        foreach ($items as $item) {
            $result['children'][] = self::computeCategoryTreeForRestrictions($item);
        }

        return $result;
    }

    /**
     * Compute category tree for content restrictions creation
     *
     * @return array
     */
    public static function computeCategoryTreeForRestrictionCreation(): array
    {
        $allCategories = BusinessCategory::allBusinessCategories([
            config(config('seeder.business_categories.affiliate.code')),
        ]);

        $groups = [];
        foreach ($allCategories as $category) {
            $groups[] = self::computeCategoryTreeForRestrictions($category);
        }

        $groupCategory = [];
        foreach ($groups as $group) {
            $groupItem = $group;
            if(count($groupItem['children']) > 0) {
                $firstGroup = $groupItem['children'][0];
                if($firstGroup['designation'] === $groupItem['designation']) {
                    $groupItem['children'] = $firstGroup['children'];
                }
            }

            $groupCategory[] = $groupItem;
        }

        return $groupCategory;
    }

    /**
     * Delete content restriction
     *
     * @return void
     */
    public function deleteRestriction(): void
    {
        $contentRestriction = $this;
        $contentRestriction->delete();
    }
}
