<?php

namespace App\Services;

use App\Models\Chat;
use App\Models\MessageDeliveryHistory;
use App\Models\MessageViewHistory;
use App\Models\RealtimeMessageAttachment;
use App\Models\RealtimeParticipant;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

trait ChatService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return Chat::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Find by uuid
    *
    * @param string $uuid
    * @return Model|Builder
    */
   public static function findByUuid(string $uuid): Model|Builder
   {
      return Chat::query()->where('uuid', $uuid)->first();
   }

   /**
    * *Find by name
    *
    * @param string $name
    * @return Model|Builder|null
    */
   public static function findByName(string $name): Model|Builder|null
   {
      return Chat::query()->where('name', $name)->first();
   }

   /**
    * Create new chat
    *
    * @param string $name
    * @param array $participantUserIds
    * @return Model|Builder
    */
   public static function createChat(string $name, array $participantUserIds = []): Model|Builder
   {
      $chat = self::store([
         'uuid' => generate_uuid('chats'),
         'name' => $name,
      ]);

      // Add participants  ...
      foreach ($participantUserIds as $participantUserId) {
         RealtimeParticipant::store([
            'chat_id' => $chat->{'id'},
            'participant_id' => $participantUserId,
         ]);
      }

      return $chat->refresh();
   }

   /**
    * Get side chat participant
    *
    * @return Model|null
    */
   public function sideParticipant(): Model|null
   {
      /** @var User $user */
      $user = auth()->user();

      $sideParticipant =
         $this
            ->participants()
            ->whereHas('user', function (Builder $builder) use ($user) {
               $builder->whereNotIn('id', [$user->{'id'}]);
            })
            ->first();

      return $sideParticipant ?? $user->{'participant'};
   }

   /**
    * Get chat history
    *
    * @param User $user
    * @return Builder[]|Collection
    */
   public static function history(User $user): Collection|array
   {
      $chatIds = RealtimeParticipant::query()->where('participant_id', $user->{'id'})->pluck('chat_id');
      $chatIds = array_unique($chatIds->toArray());

      $result = [];
      $chats = Chat::query()->with(['latestMessages.chat'])->whereIn('id', $chatIds)->get();

      foreach ($chats as $chat) {
         $messages = $chat->{'latestMessages'};
         $messagesResult = [];
         $chat['side_participant'] = $chat->sideParticipant()->load(['user.userDetail', 'user.business']);

         foreach ($messages as $message) {
            $message->{'emitter'}->load(['user.userDetail', 'user.business']);
            $message['latest_delivered_histories'] = MessageDeliveryHistory::messageDeliveryHistory($message);
            $message['latest_view_histories'] = MessageViewHistory::messageViewsHistory($message);

            // Check attachment
            /** @var RealtimeMessageAttachment $realtimeMessageAttachment */
            $realtimeMessageAttachment = $message->{'realtimeMessageAttachment'};
            if ($realtimeMessageAttachment !== null) {
               $message->unsetRelation('realtimeMessageAttachment');
               $message['realtime_message_attachment'] = $realtimeMessageAttachment->serializeForChannel();
            }

            $messagesResult[] = $message;
         }

         $chat->{'latest_messages'} = $messagesResult;
         $result[] = $chat;
      }

      return $result;
   }

   /**
    * Handle chat for new message
    *
    * @param string $name
    * @return Model|HasMany|null
    */
   public static function findByNameOrCreate(string $name): Model|HasMany|null
   {
      /** @var ?Chat $chat */
      $chat = Chat::findByName($name) ?? Chat::findByName(
         Chat::reverseChatName($name)
      );

      if ($chat === null) {
         // Create chat
         $chat = self::createChat($name);
      }

      return $chat;
   }

   /**
    * Find chat participant using user id
    *
    * @param User $user
    * @return Model|Builder|null
    */
   public function participantByUser(User $user): Model|Builder|null
   {
      return
         $this->participants()
            ->where('participant_id', $user->{'id'})
            ->first();
   }
}
