<?php

namespace App\Services;

use App\Models\CartItem;
use App\Models\CartItemAttribute;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait CartItemService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return CartItem::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return CartItem::query()->find($id);
    }

    /**
     * Create cart item
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function createCartItem(array $data): Model|Builder|null
    {
        DB::beginTransaction();
            //TODO Check user cannot add their product to cart
        try {
            /** @var User $user */
            $user = auth()->user();

            $cartItem = self::store([
                "product_quantity" => $data["product_quantity"],
                "user_id" => $user->{"id"},
                "product_id" => $data["product_id"],
            ]);

            $attributeResults = self::handleCartItemAttributes($data, $cartItem->{"id"});

            if (sizeOf($attributeResults) != config("torryme.constants.default_zero_number")) {
                if (isset($attributeResults["resultSize"]) && sizeof($attributeResults["resultSize"]) == $attributeResults["initialSize"][0]) {
                    DB::commit();
                    return $cartItem;
                } else {
                    DB::rollBack();
                    return null;
                }
            }

            DB::commit();
            $result = $cartItem;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'CartItemService::createCartItem');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Handle cart item attributes
     *
     * @param array $data
     * @param int $cartItemId
     * @return array
     */
    public static function handleCartItemAttributes(array $data, int $cartItemId): array
    {
        if (isset($data["cart_item_attributes"]) && sizeof($data["cart_item_attributes"]) != 0) {
            $initialSize = config("torryme.constants.default_zero_number");
            $cartItemAttributeArray = array();
            foreach ($data["cart_item_attributes"] as $cartItemAttribute) {
                if (isset($cartItemAttribute["product_attribute_id"]) && ProductAttribute::checkAttributeExist($cartItemAttribute["product_attribute_id"])) {
                    $cartItemAttributeArray["resultSize"][] = array(
                        CartItemAttribute::store([
                            "cart_item_id" => $cartItemId,
                            "product_attribute_id" => $cartItemAttribute["product_attribute_id"],
                        ])
                    );
                }
                $initialSize++;
            }

            // Get initial size of cart item attributes array
            $cartItemAttributeArray["initialSize"][] = $initialSize;
            return $cartItemAttributeArray;
        }

        // Return empty table in case there are no cart item attributes
        return [];
    }

    /**
     * Resume cart
     *
     * @param int|null $pageNumber
     * @param string|null $businessCategoryCode
     * @return LengthAwarePaginator
     */
    public static function resumeCart(int $pageNumber = null, string $businessCategoryCode = null): LengthAwarePaginator
    {
        /** @var User $user */
        $user = auth()->user();

        $queryBuilder = CartItem::query()->with(['user', 'product'])->where("user_id", $user->{"id"});

        if($businessCategoryCode !== null) {
            $queryBuilder = $queryBuilder->whereHas('product.businessCategory', function(Builder $builder) use($businessCategoryCode) {
                $builder->where('code', $businessCategoryCode);
            });
        }

        return
            $queryBuilder
                ->orderByDesc('created_at')
                ->paginate(
                    perPage: config("torryme.constants.items_per_page"),
                    page: $pageNumber
                );
    }

    /**
     * Update cart item
     *
     * @param array $data
     * @return null
     */
    public function updateCartItem(array $data)
    {
        DB::beginTransaction();

        try {
            //Get cart item to be updated
            $cartItem = $this;

            // Update cart item
            $updateCart = $cartItem->updateService([
                "product_quantity" => $data["product_quantity"]
            ]);

            // Check if card item has attributes
            $hasAttributes = $cartItem->{"cartAttributes"};

            // If Cart item has attributes, delete the attributes
            if ($hasAttributes != null && count($hasAttributes) > config("torryme.constants.default_zero_number")) {
                foreach ($hasAttributes as $attribute) {
                    $attribute->delete();
                }
            }

            self::handleCartItemAttributes($data, $data["cart_item_id"]);

            DB::commit();
            $result = $updateCart->refresh();
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'CartItemService::updateCartItem');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Compute cart item total price
     *
     * @return float
     */
    public function computeCartItemTotalPrice(): float
    {
        $cartItem = $this;

        // TODO Create service into ProductService to handle product price and product currency
        $productPrice = $cartItem->{"product"}->{"price"};
        $cartAttributes = $cartItem->{'cartAttributes'};
        $cartProductQuantity = $cartItem->{"product_quantity"};
        $totalProductPrice = $productPrice * $cartProductQuantity;

        $cartAttributeTotalPrice = array();
        foreach ($cartAttributes as $cartAttribute) {
            $additionalPrice = $cartAttribute->{"cartProductAttribute"} != null ? $cartAttribute->{"cartProductAttribute"}->{"additional_price"} : config("torryme.constants.default_zero_number");
            $cartAttributeTotalPrice[] = $additionalPrice;
        }

        $totalCartAttributeTotalPrice = $cartProductQuantity * array_sum($cartAttributeTotalPrice);

        return $totalProductPrice + $totalCartAttributeTotalPrice;
    }

    /**
     * Get user cart item number
     *
     * @param int $userId
     * @return int
     */
    public static function userCartItemNumber(int $userId): int
    {
        return CartItem::query()->where("user_id", $userId)->count();
    }

    /**
     * Get cart product quantity
     *
     * @param int $userId
     * @param $productId
     * @return int
     */
    public static function getCartItemQuantity(int $userId, $productId): int
    {
        return
            CartItem::query()
                ->where('user_id', $userId)
                ->where('product_id', $productId)
                ->sum('product_quantity');
    }

    /**
     * Compute cart details
     *
     * @return CartItem
     */
    public function computeCartItemDetails(): CartItem
    {
        $cartItem = $this;
        /** @var ?Product $product */
        $product = $cartItem->{"product"};
        $productDetails = $product->computeDetailsForPost();

        // Remove unused relations ...
        $cartItem->unsetRelation('user');
        $cartItem->unsetRelation('product');

        $cartItem['product'] = $productDetails;
        $cartItem['cart_attributes'] = CartItemAttribute::computeForProductCartDetails($cartItem);

        $cartItem->unsetRelation('cartAttributes');

        return $cartItem;
    }

    /**
     * Get user's cart items
     *
     * @param array $cartItemIds
     * @return \Illuminate\Support\Collection
     */
    public static function getCartItemListByBusiness(array $cartItemIds): \Illuminate\Support\Collection
    {
        return
            DB::table('cart_items')
                ->select("cart_items.id as cart_item_id", "cart_items.product_quantity as product_quantity", "cart_items.product_id as product_id", "products.business_id as business_id", "products.price_currency_id as price_currency_id")
                ->join('products', 'cart_items.product_id', '=', 'products.id')
                ->whereIn("cart_items.id", $cartItemIds)
                ->get()
                ->groupBy('business_id');
    }

    /**
     * Get user cart items
     *
     * @param User $user
     * @param Product $product
     * @return Model|Builder|null
     */
    public static function getUserCartItem(User $user, Product $product): Model|Builder|null
    {
        return CartItem::query()->where("User_id", $user->{"id"})
                                ->where("product_id", $product->{"id"})
                                ->first();
    }

}
