<?php

namespace App\Services;

use App\Exceptions\AlreadyAffiliatedToProgramException;
use App\Exceptions\ShouldBeAffiliateMarketerException;
use App\Models\AffiliateProgram;
use App\Models\Business;
use App\Models\BusinessAffiliate;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait BusinessAffiliateService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return BusinessAffiliate::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * @param int $affiliateProgramId
     * @param Business $affiliatedBusiness
     * @return Builder|Model|null
     * @throws AlreadyAffiliatedToProgramException
     * @throws ShouldBeAffiliateMarketerException
     */
    public static function affiliateToProgram(int $affiliateProgramId, Business $affiliatedBusiness): Model|Builder|null
    {
        try {
            //Check request user is an affiliate marketer
            AffiliateProgram::shouldBeAffiliateMarketer($affiliatedBusiness);

            $affiliateProgram = AffiliateProgram::findById($affiliateProgramId);

            if (self::alreadyAffiliatedToProgram($affiliateProgram->{"id"}, $affiliatedBusiness->{"id"})) {
                throw  new AlreadyAffiliatedToProgramException(__("errors.already_affiliated_to_program"));
            }

            $date = Carbon::createFromFormat(config('torryme.constants.default_date_time_format'), $affiliateProgram->{"started_at"});
            $daysToAdd = $affiliateProgram->{"active_days"};
            $expiryDate = $date->addDays($daysToAdd);

            $businessAffiliate = self::store([
                "expired_at" => $expiryDate,
                "affiliate_program_id" => $affiliateProgram->{"id"},
                "business_affiliate_id" => $affiliateProgram->{"business_id"},
                "business_affiliated_id" => $affiliatedBusiness->{"id"},
            ]);

            $result = $businessAffiliate;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessAffiliateService::affiliateToProgram');
            throw $exception;
        }

        return $result;
    }

    /**
     * Get affiliated programs
     *
     * @param Business $affiliatedBusiness
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     * @throws ShouldBeAffiliateMarketerException
     */
    public static function getAffiliatedPrograms(Business $affiliatedBusiness, int $pageNumber = null,): LengthAwarePaginator
    {
        try {
            // Check request user is an affiliate marketer
            AffiliateProgram::shouldBeAffiliateMarketer($affiliatedBusiness);
            $businessAffiliatedId = $affiliatedBusiness->{"id"};

            $queryBuilder = BusinessAffiliate::query()->with(['affiliateProgram', 'businessAffiliate']);

            $queryBuilder = $queryBuilder->where(function (Builder $builder) use ($businessAffiliatedId) {
                $builder->where("business_affiliated_id", $businessAffiliatedId);
            })->orderByDesc('created_at');

            $paginator = $queryBuilder->paginate(
                perPage: config("torryme.constants.items_per_page"),
                columns: ['*'],
                pageName: 'page',
                page: $pageNumber
            );

            $result = $paginator;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessAffiliateService::getAffiliatedPrograms');
            throw $exception;
        }

        return $result;
    }

    /**
     * Get businesses affiliated to program
     *
     * @param int $affiliateProgramId
     * @return Builder[]|Collection
     */
    public static function getProgramAffiliates(int $affiliateProgramId): Collection|array
    {
        $businessAffiliates =
            BusinessAffiliate::query()
                ->with('businessAffiliatedToProgram')
                ->where('affiliate_program_id', $affiliateProgramId)
                ->get();

        $businessAffiliatesArray = array();

        foreach ($businessAffiliates as $businessAffiliate) {
           $businessAffiliatedToProgram = $businessAffiliate->{"businessAffiliatedToProgram"};

            $businessAffiliatesArray[] = array(
                "business_affiliate_id" => $businessAffiliate->{"id"},
                "business_affiliated_id" => $businessAffiliate->{"business_affiliated_id"},
                "business_affiliated_designation" => $businessAffiliatedToProgram->{"designation"},
                "business_affiliated_user_photo" => $businessAffiliatedToProgram->{"ownerUser"}->{"avatar"},
            );
        }

        return $businessAffiliatesArray;
    }

    /**
     * Check user has already affiliated to program
     *
     * @param int $affiliateProgramId
     * @param int $businessAffiliatedId
     * @return bool
     */
    public static function alreadyAffiliatedToProgram(int $affiliateProgramId, int $businessAffiliatedId): bool
    {
        return BusinessAffiliate::query()->where(array(
            "affiliate_program_id" => $affiliateProgramId,
            "business_affiliated_id" => $businessAffiliatedId,
        ))->exists();
    }
}
