<?php

namespace App\RequestRules\Api;

use App\Enums\GameTypeEnum;
use App\Enums\MusicTypeEnum;
use App\Enums\ProductStepEnum;
use App\Models\BusinessSubCategoryChild;

class ProductRules
{
    /**
     * Create goods product rules
     *
     * @return string[]
     */
    public static function createGoods(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
        ];

        if(request()->{"step"} == ProductStepEnum::create->value) {
            $baseRules = [
                'designation' => 'required|string',
                'description' => 'required|string',
                'price' => 'required|numeric|gt:0',
                'price_currency' => 'required|string|exists:currencies,code',
                'min_order' => 'nullable|numeric|gt:0',
                'max_order' => 'nullable|numeric|gte:0',
                'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
                'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
                'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
                'hash_tags' => 'required|array|min:1',
            ];

            if (!isset(request()->{"quantity"}) && !isset(request()->{"unit"})) {
                return array_merge(
                    $baseRules,
                    ['quantity' => 'required|numeric|gt:0', 'unit' => 'required|numeric|gt:0']
                );
            }

            if (isset(request()->{"quantity"}) && request()->{"quantity"} != null) {
                return array_merge(
                    $baseRules,
                    ['quantity' => 'required|numeric|gt:0']
                );
            }

            if (isset(request()->{"unit"}) && request()->{"unit"} != null) {
                return array_merge(
                    $baseRules,
                    ['unit' => 'required|numeric|gt:0']
                );
            }
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return $baseRules;
    }

    /**
     * Create Service product rules
     *
     * @return string[]
     */
    public static function createService(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
        ];

        if(request()->{"step"} == ProductStepEnum::create->value) {
            return [
                'designation' => 'required|string',
                'description' => 'required|string',
                'quantity' => 'required|numeric|gt:0',
                'delivery_in_days' => 'required|numeric',
                'price' => 'required|numeric|gt:0',
                'price_currency' => 'required|string|exists:currencies,code',
                'min_order' => 'nullable|numeric|gt:0',
                'max_order' => 'nullable|numeric|gte:0',
                'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
                'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
                'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
                'hash_tags' => 'required|array|min:1'
            ];
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return  $baseRules;
    }

    /**
     * Create Cinema product rules
     *
     * @return array
     */
    public static function createCinema(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
        ];

        $secondBaseRule = [
            'designation' => 'required|string',
            'description' => 'required|string',
            'price_currency' => 'required|string|exists:currencies,code',
            'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
            'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
            'hash_tags' => 'required|array|min:1'
        ];

        if(request()->{"step"} == ProductStepEnum::create->value) {
            if(request()->{"business_sub_category_child_id"} != null ) {
                $businessSubCategory = BusinessSubCategoryChild::findById(request()->{"business_sub_category_child_id"});

                if($businessSubCategory != null) {
                    if($businessSubCategory->{"code"} === config('torryme.business_sub_category_child_codes.series')) {
                        return array_merge(
                            $secondBaseRule, [
                                'price' => 'required|numeric|gte:0',
                                'series_poster' => ['required', 'image', 'max:' . config('torryme.constants.max_file_size')],
                            ]
                        );
                    }

                    return  array_merge(
                        $secondBaseRule, [
                            'price' => 'required|numeric|gt:0',
                        ]
                    );
                } else {
                    return [
                        'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
                    ];
                }
            }

            return  array_merge(
                $secondBaseRule, [
                    'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
                ]
            );
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return  $baseRules;
    }

    /**
     * Create music rules
     *
     * @return array
     */
    public static function createMusic(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
            'music_type' => [
                'required',
                'in:' . implode(",", array(MusicTypeEnum::single->value, MusicTypeEnum::album->value))
            ],
        ];

        $secondBaseRule = [
            'designation' => 'required|string',
            'description' => 'required|string',
            'price' => 'required|numeric|gt:0',
            'price_currency' => 'required|string|exists:currencies,code',
            'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
            'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
            'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
            'hash_tags' => 'required|array|min:1',
        ];

        if(request()->{"step"} === ProductStepEnum::create->value) {
            if(request()->{"music_type"} === MusicTypeEnum::single->value) {
                return  $secondBaseRule;
            }

            if(request()->{"music_type"} === MusicTypeEnum::album->value) {
                return array_merge(
                    $secondBaseRule, [
                        'album_poster' => ['required', 'image', 'max:' . config('torryme.constants.max_file_size')],
                    ]
                );
            }
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return  $baseRules;
    }

    /**
     * Create library product rules
     *
     * @return array
     */
    public static function createLibrary(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
        ];

        if(request()->{"step"} == ProductStepEnum::create->value) {
            return [
                'designation' => 'required|string',
                'description' => 'required|string',
                'quantity' => 'required|numeric|gt:0',
                'price' => 'required|numeric|gt:0',
                'price_currency' => 'required|string|exists:currencies,code',
                'min_order' => 'nullable|numeric|gt:0',
                'max_order' => 'nullable|numeric|gte:0',
                'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
                'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
                'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
                'hash_tags' => 'required|array|min:1'
            ];
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return  $baseRules;
    }

    /**
     * Add game product rules
     *
     * @return string[]|\string[][]
     */
    public static function createGames(): array
    {
        $baseRules = [
            'step' => [
                'required',
                'in:' . implode(",", array(ProductStepEnum::create->value, ProductStepEnum::attachments->value))
            ],
            'game_type' => [
                'required',
                'in:' . implode(",", array(GameTypeEnum::softCopy->value, GameTypeEnum::hardCopy->value))
            ],
        ];

        $secondBaseRule = [
            'designation' => 'required|string',
            'description' => 'required|string',
            'price' => 'required|numeric|gt:0',
            'price_currency' => 'required|string|exists:currencies,code',
            'business_sub_category_id' => 'required|numeric|exists:business_sub_categories,id',
            'business_sub_category_child_id' => 'required|numeric|exists:business_sub_category_child,id',
            'business_sub_category_grand_child_id' => 'nullable|numeric|exists:business_sub_category_child,id',
            'hash_tags' => 'required|array|min:1',
        ];

        if(request()->{"step"} == ProductStepEnum::create->value) {
            if(request()->{"game_type"} === GameTypeEnum::hardCopy->value) {
                return array_merge(
                    $secondBaseRule, [
                        'quantity' => 'required|numeric|gt:0',
                        'min_order' => 'nullable|numeric|gt:0',
                        'max_order' => 'nullable|numeric|gte:0',
                        'game_cover' => ['required', 'image', 'max:' . config('torryme.constants.max_file_size')],
                    ]
                );
            }

            if(request()->{"game_type"} === GameTypeEnum::softCopy->value) {
                return $secondBaseRule;
            }
        }

        if(request()->{"step"} == ProductStepEnum::attachments->value) {
            return  [
                'product_id' => 'required|numeric|exists:products,id',
            ];
        }

        return $baseRules;
    }

    public static function createAffiliate()
    {
        return [

        ];
    }

    /**
     * Update product status rule
     *
     * @return string[]
     */
    public static function updateProductStatus(): array
    {
        return  [
            'product_id' => 'required|numeric|exists:products,id',
        ];
    }
}
