<?php

namespace App\RequestRules\Api;

use App\Enums\PostStepEnum;

class PostRules
{
   /**
    * Create post rules
    *
    * @return string[]
    */
   public static function creationRules(): array
   {
      $baseRules = [
         'step' => [
            'required',
            'in:' . implode(",", array(PostStepEnum::create->value, PostStepEnum::attachments->value))
         ],
      ];

      if (request()->{"step"} == PostStepEnum::create->value) {
         return [
            'community_id' => 'nullable|exists:communities,id',
            'product_id' => 'nullable|numeric|exists:products,id',
            'hash_tags' => 'nullable|array',
            'mentions' => 'nullable|array',
         ];
      }

      if (request()->{"step"} == PostStepEnum::attachments->value) {
         return [
            'post_id' => 'required|numeric|exists:posts,id',
         ];
      }

      return $baseRules;
   }

   /**
    * Repost rules
    *
    * @return string[]
    */
   public static function repostRules(): array
   {
      return [
         'post_id' => 'required|numeric|exists:posts,id',
      ];
   }

   /**
    * Like/Unlike post rule
    *
    * @return string[]
    */
   public static function likePost(): array
   {
      return [
         'post_id' => 'required|numeric|exists:posts,id',
      ];
   }

   /**
    * Save/UnSave post rule
    *
    * @return string[]
    */
   public static function savePost(): array
   {
      $baseRules = [
         'post_id' => 'required|numeric|exists:posts,id',
      ];

      if (isset(request()->{"favourite_category_id"})) {
         return array_merge(
            $baseRules,
            ['favourite_category_id' => 'required|numeric|exists:favourite_categories,id',]
         );
      }

      return $baseRules;
   }

   /**
    * Share/UnShare post rule
    *
    * @return string[]
    */
   public static function sharePost(): array
   {
      return [
         'post_id' => 'required|numeric|exists:posts,id',
         'description' => 'nullable|string',
         "user_ids" => "required|array|exists:users,id",
      ];
   }

   /**
    * PostComment post
    *
    * @return array
    */
   public static function commentPost(): array
   {
      if (filled(request()->{'post_comment_id'})) {
         return [
            'post_comment_id' => 'required|numeric|exists:post_comments,id',
            'comment' => 'required|string|max:250',
            'attachments' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
         ];
      }

      return [
         'post_id' => 'required|numeric|exists:posts,id',
         'comment' => 'required|string|max:250',
         'attachments' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
      ];
   }

   /**
    * Comment reply
    *
    * @return array
    */
   public static function commentReply(): array
   {
      return [
         'post_comment_id' => 'required|numeric|exists:post_comments,id',
         'comment' => 'required|string',
         'attachments' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
      ];
   }

   /**
    * Turn on comment rule
    *
    * @return string[]
    */
   public static function turnOnComment(): array
   {
      return [
         'post_id' => 'required|numeric|exists:posts,id',
      ];
   }
}
