<?php

namespace App\RequestRules\Api;

use App\Enums\CommunitySuspensionDurationEnum;
use App\Enums\CommunityVisibilityEnum;
use App\Enums\GenericStatusEnum;
use App\Enums\PostStepEnum;

class CommunityRequestRules
{
   /**
    * Create community rules
    *
    * @return array
    */
   public static function create(): array
   {
      return [
         'designation' => 'required|string|unique:communities,designation',
         'description' => 'required|string',
         'avatar' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
         'label_color' => 'nullable|string',
         'visibility' => [
            'required',
            'in:' . implode(",", array(CommunityVisibilityEnum::public->value, CommunityVisibilityEnum::private->value))
         ],
         'community_subscribers' => 'nullable|array|exists:users,id'
      ];
   }

   /**
    * Edit community rules
    *
    * @return array
    */
   public static function edit(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
         'designation' => 'nullable|string',
         'description' => 'nullable|string',
         'avatar' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
         'label_color' => 'nullable|string',
         'visibility' => [
            'nullable',
            'in:' . implode(",", array(CommunityVisibilityEnum::public->value, CommunityVisibilityEnum::private->value))
         ],
      ];
   }

   /**
    * Edit setting rules
    *
    * @return array
    */
   public static function editSetting(): array
   {

      return [
         'enable_notification' => [
            'required',
            'in:' . implode(",", array(GenericStatusEnum::enable->value, GenericStatusEnum::disable->value))
         ],
      ];
   }

   /**
    * Add community report rules
    *
    * @return string[]
    */
   public static function addReport(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
         'subject' => 'required|string',
         'message' => 'nullable|string',
      ];
   }

   /**
    * Mark as Admin or member rules
    *
    * @return string[]
    */
   public static function marksAsAdminOrMemberOrDeleteMember(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
         'user_id' => 'required|numeric|exists:users,id',
      ];
   }

   /**
    * Leave community rule
    *
    * @return string[]
    */
   public static function leaveCommunity(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
      ];
   }

   /**
    * Add suspension rules
    *
    * @return array
    */
   public static function addSuspension(): array
   {
      $baseRules = [
         'community_id' => 'required|numeric|exists:communities,id',
         'user_id' => 'required|numeric|exists:users,id',
         'suspension_duration_type' => [
            'required',
            'in:' . implode(",", array(CommunitySuspensionDurationEnum::twentyFourHours->value, CommunitySuspensionDurationEnum::sevenDays->value, CommunitySuspensionDurationEnum::fourteenDays->value, CommunitySuspensionDurationEnum::thirtyDays->value, CommunitySuspensionDurationEnum::mintyDays->value, CommunitySuspensionDurationEnum::twelveMonths->value, CommunitySuspensionDurationEnum::custom->value))
         ],
      ];

      if (request()->{"suspension_duration_type"} == CommunitySuspensionDurationEnum::custom->value) {
         return array_merge(
            $baseRules,
            [
               'suspension_duration' => ['required', 'date', 'date_format:' . config('torryme.constants.default_date_time_format')],
            ]
         );
      }

      return $baseRules;
   }

   /**
    * Remove suspension rules
    *
    * @return string[]
    */
   public static function removeSuspension(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
         'user_id' => 'required|numeric|exists:users,id',
      ];
   }

   /**
    * Repost community post
    *
    * @return array
    */
   public static function repostRules(): array
   {
      return [
         'post_id' => 'required|numeric|exists:posts,id',
      ];
   }

   /**
    * Add subscriber rule
    *
    * @return string[]
    */
   public static function addSubscriber(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
         'user_id' => 'required|numeric|exists:users,id',
      ];
   }

   /**
    * Add membership request rule
    *
    * @return string[]
    */
   public static function addMembershipRequest(): array
   {
      return [
         'community_id' => 'required|numeric|exists:communities,id',
      ];
   }
}
