<?php

namespace App\Models;

use App\Enums\OrderStatusEnum;
use App\Services\OrderService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Order extends Model
{
    use HasFactory, SoftDeletes;
    use OrderService;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'orders';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'total_customer_amount',
        'total_customer_amount_after_coupon',
        'total_supplier_amount',
        'total_supplier_amount_after_coupon',
        'type',
        'status',
        'customer_shipping_amount',
        'supplier_shipping_amount',
        'cancelled_at',
        'reason_for_cancellation',
        'delivery_date',
        'delivered_at',
        'received_at',

        'coupon_id',
        'customer_user_id',
        'supplier_business_id',
        'customer_shipping_amount_currency_id',
        'supplier_shipping_amount_currency_id',
        'customer_currency_id',
        'supplier_currency_id',
        'cancelled_by_user_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'coupon_id',
        'customer_user_id',
        'supplier_business_id',
        'customer_currency_id',
        'supplier_currency_id',
        'cancelled_by_user_id',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        "delivery_date" => "datetime",
        "cancelled_at" => "datetime",
        "delivered_at" => "datetime",
        "received_at" => "datetime",
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'created_at',
        'updated_at',
        'deleted_at',
        'delivery_date',
        'cancelled_at',
        'received_at',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['order_type'];

    /**
     * @return string|null
     */
    public function getOrderTypeAttribute(): ?string
    {
        return $this->orderRequestTypes();
    }

    /**
     * @return string|null
     */
    public function orderRequestTypes(): ?string
    {
        /** @var User $user */
        $user = auth()->user();
        $business = $user->{'business'};

        if ($business != null && $business->{"id"} === $this->{"supplier_business_id"}) {
            return OrderStatusEnum::incoming->value;
        }

        if ($user->{"id"} === $this->{"customer_user_id"}) {
            return OrderStatusEnum::outgoing->value;
        }

        return null;
    }

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /** Attributes used outside the class to create modify the resource */

    /** Relations */

    public function customerUser(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(User::class, "customer_user_id");
    }

    public function supplierBusiness(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Business::class, "supplier_business_id");
    }

    public function customerCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, "customer_currency_id");
    }

    public function customerShippingCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, "customer_shipping_amount_currency_id");
    }

    public function supplierShippingCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, "supplier_shipping_amount_currency_id");
    }

    public function supplierCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, "supplier_currency_id");
    }

    public function orderDetails(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(OrderDetail::class);
    }

    public function escrow(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(Escrow::class);
    }
}
