<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\MusicAlbumService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class MusicAlbum extends Model
{
    use HasFactory, SoftDeletes;
    Use MusicAlbumService;

    static string $prefixDir = "album";

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'music_albums';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'cover',
        'price',

        'price_currency_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'price_currency_id'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['cover_url'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /**  Append attributes */

    public function getCoverUrlAttribute(): ?string
    {
        if (filled($this->{'cover'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::musicAlbumCover,
                'file' => $this->{'cover'}
            ]);
        }

        return null;
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Build album files dir path
     *
     * @param string $path
     * @return string
     */
    public function buildAlbumFilesPath(string $path): string
    {
        $product = $this->{'product'} ?? $this->{'musicAlbum'}->{'product'};
        $ownerUserId = $product->{'business'}->{'owner_user_id'};
        $productId = $product->{'id'};

        return sprintf(
            config('torryme.paths.docs'),
            sprintf(
                '%s%s/%s%s/%s%s/%s',
                User::$prefixDir, $ownerUserId,
                Business::$prefixDir, $product->{'business_id'},
                Product::$prefixDir, $productId,
                $path
            )
        );
    }

    /**
     * Get all music album file size
     *
     * @return int
     */
    public function albumSize(): int
    {
        $totalSize = 0;
        $singles = $this->{'singles'};

        foreach ($singles as $single) {
            $totalSize = $totalSize + $single->musicSize();
        }

        return $totalSize;
    }

    /** Relations */

    public function singles(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Music::class, 'music_album_id');
    }

    public function product(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(Product::class);
    }

    public function priceCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }
}
