<?php

namespace App\Models;

use App\Enums\AttachmentTypeEnum;
use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\MusicService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class Music extends Model
{
   use HasFactory, SoftDeletes;
   use MusicService;

   /**
    * The database table used by the model.
    *
    * @var string
    */
   protected $table = 'music';

   /**
    * Attributes that should be mass-assignable.
    *
    * @var array
    */
   protected $fillable = [
      'designation',
      'description',
      'file_path',
      'cover',
      'occurrence',

      'music_album_id',
   ];

   /**
    * The attributes excluded from the model's JSON form.
    *
    * @var array
    */
   protected $hidden = [
      'music_album_id',
   ];

   /**
    * The attributes that should be casted to native types.
    *
    * @var array
    */
   protected $casts = [];

   /**
    * The attributes that should be mutated to dates.
    *
    * @var array
    */
   protected $dates = ['created_at', 'updated_at', 'deleted_at'];

   /**
    * The accessors to append to the model's array form.
    *
    * @var array
    */
   protected $appends = ['cover_url', 'music_url'];

   /**
    * The "booting" method of the model.
    *
    * @return void
    */
   public static function boot(): void
   {
      parent::boot();
   }

   /**  Append attributes */

   public function getCoverUrlAttribute(): ?string
   {
      if (filled($this->{'cover'})) {
         return route('serve.doc', [
            'resourceId' => $this->{'id'},
            'q' => StorageFileController::musicCover,
            'file' => $this->{'cover'}
         ]);
      }

      return null;
   }

   public function getMusicUrlAttribute(): ?string
   {
      if (filled($this->{'file_path'})) {
         return route('serve.doc', [
            'resourceId' => $this->{'id'},
            'q' => StorageFileController::musicFile,
            'file' => $this->{'file_path'}
         ]);
      }

      return null;
   }

   /** Attributes used outside the class to create modify the resource */

   /**
    * Build cover and music file dir path
    *
    * @param string $path
    * @return string
    */
   public function buildMusicFilesPath(string $path): string
   {
      $product = $this->{'product'} ?? $this->{'musicAlbum'}->{'product'};
      $ownerUserId = $product->{'business'}->{'owner_user_id'};
      $productId = $product->{'id'};

      if ($this->{'musicAlbum'} === null) {
         return sprintf(
            config('torryme.paths.docs'),
            sprintf(
               '%s%s/%s%s/%s%s/%s',
               User::$prefixDir, $ownerUserId,
               Business::$prefixDir, $product->{'business_id'},
               Product::$prefixDir, $productId,
               $path
            )
         );
      }

      return sprintf(
         config('torryme.paths.docs'),
         sprintf(
            '%s%s/%s%s/%s%s/%s/%s',
            User::$prefixDir, $ownerUserId,
            Business::$prefixDir, $product->{'business_id'},
            Product::$prefixDir, $productId,
            MusicAlbum::$prefixDir,
            $path
         )
      );
   }

   /**
    * Get music file size
    *
    * @return int
    */
   public function musicSize(): int
   {
      $storageOption = config('torryme.storage_option');

      if (filled($this->{'file_path'})) {
         $filePathDir = $this->buildMusicFilesPath($this->{'file_path'});

         if (Storage::disk($storageOption)->exists($filePathDir)) {
            return File::size(storage_path('app/' . $filePathDir));
         }
      }

      return 0;
   }

   /**
    * Get music duration
    *
    * @return int
    */
   public function musicDuration(): int
   {
      if (filled($this->{'file_path'})) {
         return file_duration(
            $this->buildMusicFilesPath($this->{'file_path'}),
         );
      }

      return 0;
   }

   /**
    * Music is audio or video ?
    *
    * @return string
    */
   public function musicFileType(): string
   {
      $product = $this->{'product'} ?? $this->{'musicAlbum'}->{'product'};
      if ($product->{'businessSubCategoryChild'}?->{'code'} === config('torryme.business_sub_category_child_codes.video_music')) {
         return AttachmentTypeEnum::video->value;
      }

      return AttachmentTypeEnum::audio->value;
   }

   /** Relations */

   public function product(): \Illuminate\Database\Eloquent\Relations\HasOne
   {
      return $this->hasOne(Product::class);
   }

   public function currency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Currency::class);
   }

   public function musicAlbum(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(MusicAlbum::class, 'music_album_id');
   }
}
