<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\MovieService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class Movie extends Model
{
    use HasFactory, SoftDeletes;
    use MovieService;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'movies';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'cover',
        'file_path',
        'price',

        'price_currency_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'price_currency_id'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['cover_url', 'movie_url'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /**  Append attributes */

    public function getCoverUrlAttribute(): ?string
    {
        if (filled($this->{'cover'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::movieCover,
                'file' => $this->{'cover'}
            ]);
        }

        return null;
    }

    public function getMovieUrlAttribute(): ?string
    {
        if (filled($this->{'file_path'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::movieFile,
                'file' => $this->{'file_path'}
            ]);
        }

        return null;
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Build cover and movie file dir path
     *
     * @param string $path
     * @return string
     */
    public function buildMovieFilesPath(string $path): string
    {
        $product = $this->{'product'};
        $ownerUserId = $product->{'business'}->{'owner_user_id'};
        $productId = $product->{'id'};

        return sprintf(
            config('torryme.paths.docs'),
            sprintf(
                '%s%s/%s%s/%s%s/%s',
                User::$prefixDir, $ownerUserId,
                Business::$prefixDir, $product->{'business_id'},
                Product::$prefixDir, $productId,
                $path
            )
        );
    }

    /**
     * Get movie file size
     *
     * @return int
     */
    public function movieFileSize(): int
    {
        $storageOption = config('torryme.storage_option');

        if(filled($this->{'file_path'})) {
            $filePathDir = $this->buildMovieFilesPath($this->{'file_path'});

            if(Storage::disk($storageOption)->exists($filePathDir)) {
                return File::size(storage_path('app/' . $filePathDir));
            }
        }

        return 0;
    }

    /**
     * Get movie duration
     *
     * @return int
     */
    public function movieDuration(): int
    {
        if(filled($this->{'file_path'})) {
            return file_duration(
                $this->buildMovieFilesPath($this->{'file_path'}),
            );
        }

        return 0;
    }

    /** Relations */

    public function product(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(Product::class);
    }

    public function priceCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }
}
