<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\CommunityService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Community extends Model
{
   use HasFactory;
   use SoftDeletes;
   use CommunityService;

   static string $prefixDir = "communities/community_";

   /**
    * The database table used by the model.
    *
    * @var string
    */
   protected $table = 'communities';

   /**
    * Attributes that should be mass-assignable.
    *
    * @var array
    */
   protected $fillable = [
      'designation',
      'description',
      'uuid',
      'avatar',
      'label_color',
      'visibility',

      'create_by_user_id',
   ];

   /**
    * The attributes excluded from the model's JSON form.
    *
    * @var array
    */
   protected $hidden = [
      'create_by_user_id',
   ];

   /**
    * The attributes that should be casted to native types.
    *
    * @var array
    */
   protected $casts = [];

   /**
    * The attributes that should be mutated to dates.
    *
    * @var array
    */
   protected $dates = ['created_at', 'updated_at', 'deleted_at'];

   /**
    * The accessors to append to the model's array form.
    *
    * @var array
    */
   protected $appends = ['number_of_subscribers', 'community_avatar', 'last_membership_request', 'community_url'];

   /**
    * @return mixed
    */
   public function getLastMembershipRequestAttribute(): mixed
   {
      return CommunityMembershipRequest::getLastMembershipRequest($this->{"id"});
   }

   /**
    * @return int
    */
   public function getNumberOfSubscribersAttribute(): int
   {
      return CommunitySubscriber::numberOfSubscribers($this->{"id"});
   }

   /**
    * @return string|null
    */
   public function getCommunityAvatarAttribute(): ?string
   {
      return $this->computeAvatarLink();
   }

   /**
    * Compute avatar url
    *
    * @return string|null
    */
   public function computeAvatarLink(): ?string
   {
      if (filled($this->{'avatar'})) {
         return build_storage_file_url($this, StorageFileController::communityFlag, $this->{'avatar'});
      }

      return null;
   }

   /**
    * Build all community files path
    *
    * @param string $fileName
    * @return string|null
    */
   public function buildCommunityFilesPath(string $fileName): ?string
   {
      $creatorId = $this->{'create_by_user_id'};

      return sprintf(
         config('torryme.paths.docs'),
         sprintf(
            '%s%s/%s%s/%s',
            User::$prefixDir, $creatorId,
            self::$prefixDir, $this->{'id'},
            $fileName
         )
      );
   }

   /**
    * Build community url
    *
    * @return string
    */
   public function getCommunityUrlAttribute(): string
   {
      return build_community_url($this->{"id"});
   }

   /**
    * The "booting" method of the model.
    *
    * @return void
    */
   public static function boot(): void
   {
      parent::boot();
   }

   /** Attributes used outside the class to create modify the resource */

   /** Relations */
   public function communitySubscribers(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(CommunitySubscriber::class);
   }

   public function creator(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(User::class, 'create_by_user_id');
   }

   public function communityMembershipRequests(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(CommunityMembershipRequest::class);
   }

   public function communitySetting(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(CommunitySetting::class);
   }

   public function communityReport(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(CommunityReport::class);
   }

   public function communitySuspension(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(CommunitySuspension::class);
   }

   public function communityPost(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(Post::class);
   }

}
