<?php

namespace App\Models;

use App\Enums\UserVerificationStatusEnum;
use App\Services\BusinessService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Business extends Model
{
    use HasFactory, SoftDeletes;
     use BusinessService;

    static string $prefixDir = "business_";

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'businesses';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'geographic_location',
        'address',
        'description',

        'owner_user_id',
        'business_category_id',
        'country_id',
        'city_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'owner_user_id',
        'business_category_id',
        'country_id',
        'city_id',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['avatar', 'user_name', 'business_parent'];

    /**  Append attributes */

    public function getBusinessParentAttribute(): ?string
    {
        if($this->isAffiliate() ) {
            return $this->lastAffiliation();
        }

        return null;
    }

    public function getAvatarAttribute(): ?string
    {
        return $this->{'ownerUser'}->{'avatar'};
    }

    public function getUserNameAttribute(): ?string
    {
        return $this->{'ownerUser'}->{'user_name'};
    }

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Full path to product assets
     *
     * @param int $productId
     * @return string
     */
    public function buildProductDir(int $productId): string
    {
        return sprintf(
            User::$prefixDir . "%s/" . self::$prefixDir . "%s/" . Product::$prefixDir . "%s",
            $this->{'owner_user_id'},
            $this->{'id'},
            $productId,
        );
    }

    /** Relations */

    public function businessCategory(): BelongsTo
    {
        return $this->belongsTo(BusinessCategory::class);
    }

    public function ownerUser(): BelongsTo
    {
        return $this->belongsTo(User::class, "owner_user_id");
    }

    public function businessProduct(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Product::class, 'business_id');
    }
}
