<?php

namespace App\Models;

use App\Enums\AttachmentTypeEnum;
use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\BookService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class Book extends Model
{
    use HasFactory, SoftDeletes;
    use BookService;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'books';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'cover',
        'file_path',
        'price',

        'price_currency_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'price_currency_id'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['cover_url', 'book_url'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /**  Append attributes */

    public function getCoverUrlAttribute(): ?string
    {
        if (filled($this->{'cover'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::bookCover,
                'file' => $this->{'cover'}
            ]);
        }

        return null;
    }

    public function getBookUrlAttribute(): ?string
    {
        if (filled($this->{'file_path'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::bookFile,
                'file' => $this->{'file_path'}
            ]);
        }

        return null;
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Build cover or book file dir path
     *
     * @param string $path
     * @return string|null
     */
    public function buildBookFilesPath(string $path): ?string
    {
        $product = $this->{'product'};
        $ownerUserId = $product->{'business'}->{'owner_user_id'};
        $productId = $product->{'id'};

        return sprintf(
            config('torryme.paths.docs'),
            sprintf(
                '%s%s/%s%s/%s%s/%s',
                User::$prefixDir, $ownerUserId,
                Business::$prefixDir, $product->{'business_id'},
                Product::$prefixDir, $productId,
                $path
            )
        );
    }

    /**
     * Get book file size
     *
     * @return int
     */
    public function bookSize(): int
    {
        $storageOption = config('torryme.storage_option');

        if(filled($this->{'file_path'})) {
            $filePathDir = $this->buildBookFilesPath($this->{'file_path'});

            if(Storage::disk($storageOption)->exists($filePathDir)) {
                return File::size(storage_path('app/' . $filePathDir));
            }
        }

        return 0;
    }

    /**
     * Ebook, audio or hard copy ?
     *
     * @return ?string
     */
    public function bookFileType(): ?string
    {
        if($this->{'product'}->{'businessSubCategoryChild'}?->{'code'} === config('torryme.business_sub_category_child_codes.ebook')) {
            return AttachmentTypeEnum::pdf->value;
        }

        if($this->{'product'}->{'businessSubCategoryChild'}?->{'code'} === config('torryme.business_sub_category_child_codes.audio_book')) {
            return AttachmentTypeEnum::audio->value;
        }

        return null;
    }

    /**
     * If the book is an ebook, we try to calculate its number of pages
     *
     * @return int|null
     */
    public function totalPage(): ?int
    {
        if(filled($this->{'file_path'})) {
            $filePathDir = $this->buildBookFilesPath($this->{'file_path'});

            if(Storage::disk(config('torryme.storage_option'))->exists($filePathDir)) {
                return count_pdf_page(
                    storage_path('app/' . $filePathDir)
                );
            }
        }

        return null;
    }

    /** Relations */

    public function product(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(Product::class);
    }

    public function priceCurrency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }
}
