<?php

namespace App\Http\Controllers\Api\Platform;

use App\Http\Controllers\Controller;
use App\Models\CartItem;
use App\RequestRules\Api\CartItemRequestRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class CartItemController extends Controller
{
    /**
     * Get auth user cart status
     *
     * @return Application|ResponseFactory|Response
     */
    public function status(): Response|Application|ResponseFactory
    {
        $authUser = auth()->user();

        try {
            return api_response(
                100,
                "Okay",
                CartItem::userCartItemNumber($authUser->{'id'})
            );
        } catch (\Exception $exception) {
            log_debug($exception, "CartItemController::create");
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Create cart item
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function create(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(CartItemRequestRules::create());
        $data = $request->all();

        try {
            /** @var CartItem $cartItemResult */
            $cartItemResult = CartItem::createCartItem($data);

            if ($cartItemResult !== null) {
                return api_response(100, "Okay", $cartItemResult->computeCartItemDetails());
            }
        } catch (\Exception $exception) {
            log_debug($exception, "CartItemController::create");
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Resume cart activity
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function resume(Request $request): Response|Application|ResponseFactory
    {
        try {
            $resumeCart = CartItem::resumeCart(
                $request->get("page"),
                $request->get("category"),
            );

            $cartItemArray = array();
            foreach ($resumeCart as $cart) {
                $cartItem = $cart->computeCartItemDetails();
                if($cartItem->{'product'} !== null) {
                    $cartItemArray[] = $cartItem;
                }
            }

            $finalData = default_paginator_format(
                $resumeCart->lastPage(),
                $resumeCart->total(),
                $resumeCart->currentPage(),
                "cart_items",
                $cartItemArray,
            );

            return api_response(100, "Okay", $finalData);
        } catch (\Exception $exception) {
            log_debug($exception, "CartItemController::resume");
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Update cart item
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function update(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(CartItemRequestRules::update());
        $data = $request->all();

        try {
            /** @var CartItem $cartItem */
            $cartItem = CartItem::findById($data["cart_item_id"]);

            $cartItemUpdate = $cartItem->updateCartItem($data);

            if ($cartItemUpdate != null) {
                return api_response(100, "Okay", $cartItemUpdate->computeCartItemDetails());
            }
        } catch (\Exception $exception) {
            log_debug($exception, "CartItemController::update");
        }

        return api_response(120, __("errors.unknown_error"));
    }
}
