<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\ShouldBeAffiliateMarketerException;
use App\Exceptions\ShouldNotBeAffiliateMarketerException;
use App\Http\Controllers\Controller;
use App\Models\AffiliateProgram;
use App\Models\BusinessAffiliate;
use App\Models\User;
use App\RequestRules\Api\AffiliateProgramRules;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class AffiliateProgramController extends Controller
{
    /**
     * Get business programs with affiliates
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function businessAffiliates(Request $request): Response|Application|ResponseFactory
    {
        try {
            /** @var User $user */
            $user = auth()->user();
            $business = $user->{"business"};

            $businessProgramsWithAffiliates = AffiliateProgram::businessProgramsWithAffiliates($business, $request->get("page"));

            $totalPages = $businessProgramsWithAffiliates->total() == config("torryme.constants.total_pages_for_no_transactions") ? config("torryme.constants.total_pages_for_no_transactions") : $businessProgramsWithAffiliates->lastPage();

            $affiliateProgramsArray = array();
            foreach ($businessProgramsWithAffiliates as $affiliateProgram) {
                $affiliateProgramsArray[] = array(
                    'program' => $affiliateProgram,
                    'business_affiliates' => BusinessAffiliate::getProgramAffiliates($affiliateProgram->{"id"})
                );
            }

            $finalData = default_paginator_format(
                $businessProgramsWithAffiliates->lastPage(),
                $totalPages,
                $businessProgramsWithAffiliates->currentPage(),
                'business_affiliates',
                $affiliateProgramsArray
            );

            return api_response(100, "Okay", $finalData);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramController::businessAffiliates');
            $affiliateProgramExceptions = self::affiliateProgramExceptions($exception);
            if ($affiliateProgramExceptions != null) {
                return $affiliateProgramExceptions;
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Create affiliate program
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function createProgram(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(AffiliateProgramRules::create());
        $data = $request->all();
        try {
            /** @var User $user */
            $user = auth()->user();
            $business = $user->{"business"};

            $affiliateProgram = AffiliateProgram::createProgram($data, $business);

            return api_response(100, "Okay", $affiliateProgram);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramController::createProgram');

            $affiliateProgramExceptions = self::affiliateProgramExceptions($exception);
            if ($affiliateProgramExceptions != null) {
                return $affiliateProgramExceptions;
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Get all available programs
     *
     * @param Request $request
     * @return array|Application|ResponseFactory|Response
     */
    public function availablePrograms(Request $request): Response|array|Application|ResponseFactory
    {
        try {
            /** @var User $user */
            $user = auth()->user();
            $affiliatedBusiness = $user->{"business"};

            $availablePrograms = AffiliateProgram::allValidPrograms($affiliatedBusiness, $request->get("page"));

            $arrayResults = array();
            foreach ($availablePrograms as $availableProgram) {
                $arrayResults[] = $availableProgram->load('business');
            }

            $finalData = default_paginator_format(
                $availablePrograms->lastPage(),
                $availablePrograms->total(),
                $availablePrograms->currentPage(),
                'programs',
                $arrayResults
            );

            return api_response(100, "Okay", $finalData);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramController::availablePrograms');

            $affiliateProgramExceptions = self::affiliateProgramExceptions($exception);
            if ($affiliateProgramExceptions != null) {
                return $affiliateProgramExceptions;
            }

        }

        return api_response(120, __("errors.unknown_error"));
    }

    // Controller utilities

    /**
     * Return exceptions
     *
     * @param Exception $exception
     * @return Application|ResponseFactory|Response|null
     */
    protected static function affiliateProgramExceptions(Exception $exception): Response|Application|ResponseFactory|null
    {
        if ($exception instanceof ShouldNotBeAffiliateMarketerException || $exception instanceof ShouldBeAffiliateMarketerException) {
            return api_response(120, $exception->getMessage());
        }

        return null;
    }

}
