<?php

namespace App\Http\Controllers\Api\Platform;

use App\Enums\GenericStatusEnum;
use App\Http\Controllers\Controller;
use App\Models\AccountPrivacySetting;
use App\Models\User;
use App\RequestRules\Api\AccountPrivacySettingRequestRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class AccountPrivacySettingsController extends Controller
{
   /**
    * User account privacy settings
    *
    * @return Application|ResponseFactory|Response
    */
   public function index(): Response|Application|ResponseFactory
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         return api_response(100, 'Ok', $user->{"accountPrivacySetting"});
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'AccountPrivacySettingsController::index');
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Edit account privacy setting
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function edit(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(AccountPrivacySettingRequestRules::editAccountPrivacy());
      $data = $request->all();

      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var AccountPrivacySetting $accountPrivacy */
         $accountPrivacy = $user->{"accountPrivacySetting"};

         $results = $accountPrivacy->updatePrivacySettings($data);

         return api_response(100, 'Ok', $results);
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'AccountPrivacySettingsController::edit');
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Enable or disable 2factor auth
    *
    * @return Application|ResponseFactory|Response
    */
   public function enableOrDisable2factorAuth(): Response|Application|ResponseFactory
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var AccountPrivacySetting $accountPrivacy */
         $accountPrivacy = $user->{"accountPrivacySetting"};

         $results = $accountPrivacy->updatePrivacySettings([
            'enable_2factor_auth' => $accountPrivacy->{'enable_2factor_auth'} === GenericStatusEnum::enable->value ? GenericStatusEnum::disable->value : GenericStatusEnum::enable->value,
         ]);

         if(count($user->verifiedEmails()) <= 0) {
            return api_response(120, 'Ok', $results);
         }

         return api_response(100, 'Ok', $results);
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'AccountPrivacySettingsController::edit');
      }

      return api_response(101, __('errors.unknown_error'));
   }
}
