<?php

namespace App\Http\Controllers\Api\Auth;

use App\Exceptions\UserDeviceAlreadyExistException;
use App\Models\AccountPrivacySetting;
use App\Models\User;
use App\Models\UserDevice;
use App\RequestRules\Api\UserRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;

class LoginController extends AuthenticateController
{
   /**
    * Try to authenticate user
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    * @throws ValidationException
    */
   public function authenticate(Request $request): Response|Application|ResponseFactory
   {
      $this->validate($request, UserRules::authenticationRules());
      $username = $request->{'telephone'};

      try {
         /** @var $user User */
         $user = User::oneStepAuthentication($username);

         if ($user === null) {
            return api_response(101, __('errors.unknown_user'));
         }

         if ($user->verifyPassword($request->{'password'})) {
            // Try to check device, create or replace it
            $this->handleUserDevice($user, $request->all());

            // Check user 2factor auth
            if($user->twoFactorAuthIsEnable()) {
               return $this->respondWithToken(
                  $user->refresh(),
                  code: 140,
                  data: ['user' => self::responseWithUser($user)]
               );
            }

            return $this->respondWithToken(
               $user->refresh(),
               data: ['user' => self::responseWithUser($user)]
            );
         } else {
            return api_response(101, __('errors.unknown_user_with_password'));
         }
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'LoginController::authenticate');

         if ($exception instanceof UserDeviceAlreadyExistException) {
            return $this->respondWithToken(
               $user->refresh(),
               code: 102,
               message: __('errors.another_session_is_opened', ['device' => $user->{'userDevice'}->fullName()]),
               data: ['device' => $user->{'userDevice'}]
            ); // New user device ? Request user approbation
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /** Utilities */

   /**
    * Managing devices during user connection
    *
    * @param User $user
    * @param array $data
    * @return mixed
    * @throws UserDeviceAlreadyExistException
    */
   public function handleUserDevice(User $user, array $data): mixed
   {
      if ($user->{'userDevice'} === null) {
         $deviceData = Arr::only($data, UserDevice::creationAttributes());
         $deviceData['user_id'] = $user->{'id'};

         return UserDevice::store($deviceData);
      }

      if (!$user->{'userDevice'}->hasSameDeviceId($data['native_device_id'])) {
         throw new UserDeviceAlreadyExistException(); // New user device ? Request user approbation
      }

      return $user->{'userDevice'}->updateService(
         Arr::only($data, UserDevice::updateAttributes()),
      );
   }
}
