<?php

namespace App\Concerns;

use App\Enums\AttachmentTypeEnum;
use App\Enums\ProductStatusEnum;
use App\Models\Book;
use App\Models\Business;
use App\Models\CartItem;
use App\Models\Movie;
use App\Models\Music;
use App\Models\MusicAlbum;
use App\Models\Post;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\Series;
use App\Models\SeriesSeason;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

trait ProductShopConcern
{
   /**
    * Get business products sample
    *
    * @param Business $business
    * @param int|null $page
    * @return array
    */
   public static function businessProductSample(Business $business, int $page = null): array
   {
      $products = [];
      $paginator = null;
      $queryBuilder =
         Product::query()
            ->where('business_id', $business->{'id'})
            ->where('business_category_id', $business->{'business_category_id'})
            ->whereNull('activated_at');

      // Goods
      if ($business->isGoodsBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->serializeSampleAttributes();
         }
      }

      // Services
      if ($business->isServiceBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->serializeSampleAttributes();
         }
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $paginator =
            $queryBuilder
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('series_id')->orWhereNotNull('movie_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $movie = $product->{'movie'};
            $series = $product->{'series'};
            $productData = $product->serializeSampleAttributes();

            if ($movie !== null) {
               $productData['designation'] = $movie->{'designation'};
            }

            if ($series !== null) {
               $productData['designation'] = $series->{'designation'};
            }
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $paginator =
            $queryBuilder
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('music_id')->orWhereNotNull('music_album_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $music = $product->{'music'};
            $musicAlbum = $product->{'musicAlbum'};
            $productData = $product->serializeSampleAttributes();

            if ($music !== null) {
               $productData['designation'] = $music->{'designation'};
            }

            if ($musicAlbum !== null) {
               $productData['designation'] = $musicAlbum->{'designation'};
            }
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $paginator =
            $queryBuilder
               ->whereNotNull('book_id')
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $book = $product->{'book'};
            $productData = $product->serializeSampleAttributes();

            if ($book !== null) {
               $productData['designation'] = $book->{'designation'};
            }
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }

      if ($paginator === null) return [];

      return array(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         $products,
      );
   }

   /**
    * Compute product details for order details
    *
    * @return array
    */
   public function computeDetailsForOrderDetails(): array
   {
      $data = $this->serializeCommonAttributes(withCartInfo: false);
      $business = $this->{'business'};

      // Goods
      if ($business->isGoodsBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForGoodsProduct(),
         );
      }

      // Services
      if ($business->isServiceBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForProductService(),
         );
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $movie = $this->{'movie'};
         $series = $this->{'series'};

         if ($movie !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductMovie($movie),
            );
         } else if ($series !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductSeries($series),
            );
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $music = $this->{'music'};
         $musicAlbum = $this->{'musicAlbum'};

         if ($music !== null) {
            $data = array(
               ...$this->computeSampleAttributeForMusicProduct($music),
            );
         } else if ($musicAlbum !== null) {
            $data = array(
               ...$this->computeSampleAttributeForAlbumMusicProduct($musicAlbum),
            );
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $book = $this->{'book'};

         if ($book !== null) {
            $data = array(
               ...$this->computeSampleAttributeForBookProduct($book),
            );
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }


      return $data;
   }

   /**
    * Compute product details for post
    *
    * @return array|null
    */
   public function computeDetailsForPost(): ?array
   {
      $data = [];
      $business = $this->{'business'};

      // Goods
      if ($business->isGoodsBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForGoodsProduct(),
            "rating" => $this->averageRating(),
         );
      }

      // Services
      if ($business->isServiceBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForProductService(),
            "rating" => $this->averageRating(),
         );
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $movie = $this->{'movie'};
         $series = $this->{'series'};

         if ($movie !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductMovie($movie),
               "rating" => $this->averageRating(),
            );
         } else if ($series !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductSeries($series),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $music = $this->{'music'};
         $musicAlbum = $this->{'musicAlbum'};

         if ($music !== null) {
            $data = array(
               ...$this->computeSampleAttributeForMusicProduct($music),
               "rating" => $this->averageRating(),
            );
         } else if ($musicAlbum !== null) {
            $data = array(
               ...$this->computeSampleAttributeForAlbumMusicProduct($musicAlbum),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $book = $this->{'book'};

         if ($book !== null) {
            $data = array(
               ...$this->computeSampleAttributeForBookProduct($book),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }

      if (!filled($data)) {
         return null;
      }

      return $data;
   }

   /**
    * Compute product details for visitor
    *
    * @return array|null
    */
   public function computeDetailsForVisitor(): ?array
   {
      $data = [];
      $user = auth()->user();
      $business = $this->{'business'};

      // Goods
      if ($business->isGoodsBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForGoodsProduct(),
            "total_opinions" => $this->totalRating(),
            "rating" => $this->averageRating(),
            "total_rating" => $this->totalRating(),
            "similar_products" => [],
            "liked" => $this->userLiked($user) ? 1 : 0,
            "saved" => $this->userSaved($user) ? 1 : 0,
         );
      }

      // Services
      if ($business->isServiceBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForProductService(),
            "total_opinions" => $this->totalRating(),
            "rating" => $this->averageRating(),
            "total_rating" => $this->totalRating(),
            "similar_products" => [],
            "liked" => $this->userLiked($user) ? 1 : 0,
            "saved" => $this->userSaved($user) ? 1 : 0,
         );
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $movie = $this->{'movie'};
         $series = $this->{'series'};

         if ($movie !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductMovie($movie),
               "total_opinions" => $this->totalRating(),
               "rating" => $this->averageRating(),
               "total_rating" => $this->totalRating(),
               "similar_products" => [],
               "liked" => $this->userLiked($user) ? 1 : 0,
               "saved" => $this->userSaved($user) ? 1 : 0,
            );
         } else if ($series !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductSeries($series, [], true),
               "total_opinions" => $this->totalRating(),
               "rating" => $this->averageRating(),
               "total_rating" => $this->totalRating(),
               "similar_products" => [],
               "liked" => $this->userLiked($user) ? 1 : 0,
               "saved" => $this->userSaved($user) ? 1 : 0,
            );
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $music = $this->{'music'};
         $musicAlbum = $this->{'musicAlbum'};

         if ($music !== null) {
            $data = array(
               ...$this->computeSampleAttributeForMusicProduct($music),
               "total_opinions" => $this->totalRating(),
               "rating" => $this->averageRating(),
               "total_rating" => $this->totalRating(),
               "similar_products" => [],
               "liked" => $this->userLiked($user) ? 1 : 0,
               "saved" => $this->userSaved($user) ? 1 : 0,
            );
         } else if ($musicAlbum !== null) {
            $data = array(
               ...$this->computeSampleAttributeForAlbumMusicProduct($musicAlbum),
               "total_opinions" => $this->totalRating(),
               "rating" => $this->averageRating(),
               "total_rating" => $this->totalRating(),
               "similar_products" => [],
               "liked" => $this->userLiked($user) ? 1 : 0,
               "saved" => $this->userSaved($user) ? 1 : 0,
            );
            $data['album']['singles'] = Music::computeSampleDetails($musicAlbum->{'singles'});
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $book = $this->{'book'};

         if ($book !== null) {
            $data = array(
               ...$this->computeSampleAttributeForBookProduct($book),
               "total_opinions" => $this->totalRating(),
               "rating" => $this->averageRating(),
               "total_rating" => $this->totalRating(),
               "similar_products" => [],
               "liked" => $this->userLiked($user) ? 1 : 0,
               "saved" => $this->userSaved($user) ? 1 : 0,
            );
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }

      if (!filled($data)) {
         return null;
      }

      return $data;
   }

   /**
    * Compute product details for reviews
    *
    * @return array|null
    */
   public function computeDetailsForReviews(): ?array
   {
      $data = [];
      $business = $this->{'business'};

      // Goods
      if ($business->isGoodsBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForGoodsProduct(),
            "rating" => $this->averageRating(),
         );
      }

      // Services
      if ($business->isServiceBusiness()) {
         $data = array(
            ...$this->computeSampleAttributeForProductService(),
            "rating" => $this->averageRating(),
         );
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $movie = $this->{'movie'};
         $series = $this->{'series'};

         if ($movie !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductMovie($movie),
               "rating" => $this->averageRating(),
            );
         } else if ($series !== null) {
            $data = array(
               ...$this->computeSampleAttributeForProductSeries($series),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $music = $this->{'music'};
         $musicAlbum = $this->{'musicAlbum'};

         if ($music !== null) {
            $data = array(
               ...$this->computeSampleAttributeForMusicProduct($music),
               "rating" => $this->averageRating(),
            );
         } else if ($musicAlbum !== null) {
            $data = array(
               ...$this->computeSampleAttributeForAlbumMusicProduct($musicAlbum),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $book = $this->{'book'};

         if ($book !== null) {
            $data = array(
               ...$this->computeSampleAttributeForBookProduct($book),
               "rating" => $this->averageRating(),
            );
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }

      if (!filled($data)) {
         return null;
      }

      return $data;
   }

   /**
    * We try to calculate and return the products available in the store of a business
    *
    * @param Business $business
    * @param int|null $page
    * @return array
    */
   public static function productsForBusinessShop(Business $business, int $page = null): array
   {
      $products = [];
      $paginator = null;
      $queryBuilder =
         Product::query()
            ->where('business_id', $business->{'id'})
            ->where('business_category_id', $business->{'business_category_id'})
            ->whereNull('activated_at');

      // Goods
      if ($business->isGoodsBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->computeSampleAttributeForGoodsProduct([
               "total_post" => Post::totalProductPost($product),
            ]);
         }
      }

      // Services
      if ($business->isServiceBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->computeSampleAttributeForProductService([
               "total_post" => Post::totalProductPost($product),
            ]);
         }
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $paginator =
            $queryBuilder
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('series_id')->orWhereNotNull('movie_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $movie = $product->{'movie'};
            $series = $product->{'series'};

            if ($movie !== null) {
               $products[] = $product->computeSampleAttributeForProductMovie($movie, [
                  "total_post" => Post::totalProductPost($product),
               ]);
            } else if ($series !== null) {
               // Build generic data
               $products[] = $product->computeSampleAttributeForProductSeries($series, [
                  "total_post" => Post::totalProductPost($product),
               ]);
            } else {
               // At this level, we don't have a movie, we don't have
               // a series, so the product is incomplete.
               $products[] = array(
                  ...$product->serializeCommonAttributes(),
               );
            }
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         $paginator =
            $queryBuilder
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('music_id')->orWhereNotNull('music_album_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $music = $product->{'music'};
            $musicAlbum = $product->{'musicAlbum'};

            if ($music !== null) {
               $products[] = $product->computeSampleAttributeForMusicProduct($music, [
                  "total_post" => Post::totalProductPost($product),
               ]);
            } else if ($musicAlbum !== null) {
               $products[] = $product->computeSampleAttributeForAlbumMusicProduct($musicAlbum, [
                  "total_post" => Post::totalProductPost($product),
               ]);
            } else {
               // At this level, we don't have a music, we don't have
               // the album music, so the product is incomplete.
               $products[] = array(
                  ...$product->serializeCommonAttributes(),
               );
            }
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $paginator =
            $queryBuilder
               ->whereNotNull('book_id')
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $book = $product->{'book'};

            if ($book !== null) {
               $products[] = $product->computeSampleAttributeForBookProduct($book, [
                  "total_post" => Post::totalProductPost($product),
               ]);
            } else {
               // At this level, we don't have book so the product is incomplete.
               $products[] = array(
                  ...$product->serializeCommonAttributes(),
               );
            }
         }
      }

      // Game
      if ($business->isGameBusiness()) {
         // ...
      }

      if ($paginator === null) return [];

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         "products",
         $products,
      );
   }

   /**
    * We try to calculate and return the products available in the store of a business for visitor
    *
    * @param Business $business
    * @param int|null $page
    * @return array
    */
   public static function productsForVisitorBusinessShop(Business $business, int $page = null): array
   {
      $products = [];
      $paginator = null;
      $user = auth()->user();

      $queryBuilder =
         Product::query()
            ->where('business_id', $business->{'id'})
            ->where('business_category_id', $business->{'business_category_id'})
            ->where('creation_status', ProductStatusEnum::complete->value)
            ->whereNull('activated_at');

      // Goods
      if ($business->isGoodsBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->computeSampleAttributeForGoodsProduct(array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         }
      }

      // Services
      if ($business->isServiceBusiness()) {
         $paginator = self::goodsOrServiceProductQueryBuilder($queryBuilder, $page);

         foreach ($paginator->items() as $product) {
            $products[] = $product->computeSampleAttributeForProductService(array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         }
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         $seriesPaginator =
            Product::query()
               ->where('business_id', $business->{'id'})
               ->where('business_category_id', $business->{'business_category_id'})
               ->where('creation_status', ProductStatusEnum::complete->value)
               ->whereNull('activated_at')
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('series_id')->whereNull('movie_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         $moviesPaginator =
            Product::query()
               ->where('business_id', $business->{'id'})
               ->where('business_category_id', $business->{'business_category_id'})
               ->where('creation_status', ProductStatusEnum::complete->value)
               ->whereNull('activated_at')
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('movie_id')->whereNull('series_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );


         return array(
            "movies" => self::computeCinemaShopForVisitor($moviesPaginator),
            "series" => self::computeCinemaShopForVisitor($seriesPaginator),
         );
      }

      // Music
      if ($business->isMusicBusiness()) {
         $singlePaginator =
            Product::query()
               ->where('business_id', $business->{'id'})
               ->where('business_category_id', $business->{'business_category_id'})
               ->where('creation_status', ProductStatusEnum::complete->value)
               ->whereNull('activated_at')
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('music_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         $albumPaginator =
            Product::query()
               ->where('business_id', $business->{'id'})
               ->where('business_category_id', $business->{'business_category_id'})
               ->where('creation_status', ProductStatusEnum::complete->value)
               ->whereNull('activated_at')
               ->where(function (Builder $builder) {
                  $builder->whereNotNull('music_album_id');
               })
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );


         return array(
            "singles" => self::computeMusicShopForVisitor($singlePaginator),
            "albums" => self::computeMusicShopForVisitor($albumPaginator),
         );
      }

      // Library
      if ($business->isLibraryBusiness()) {
         $paginator =
            $queryBuilder
               ->whereNotNull('book_id')
               ->orderByDesc('created_at')->paginate(
                  perPage: config('torryme.constants.items_per_page'),
                  page: $page,
               );

         foreach ($paginator->items() as $product) {
            $book = $product->{'book'};

            if ($book !== null) {
               $products[] = $product->computeSampleAttributeForBookProduct($book, array(
                  "rating" => $product->averageRating(),
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            } else {
               // At this level, we don't have book so the product is incomplete.
               $products[] = array(
                  ...$product->serializeCommonAttributes(),
               );
            }
         }
      }

      if ($page === null) {
         return $products;
      }

      if ($paginator === null) return [];

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         "products",
         $products,
      );
   }

   /**
    * Calculation of similar products
    *
    * @param array $excludeIds
    * @param int|null $page
    * @return array
    */
   public function similarProductsToAnother(array $excludeIds = [], int $page = null): array
   {
      $products = [];
      $paginator = null;
      $product = $this;

      $business = $product->{'business'};
      $user = auth()->user();

      $queryBuilder =
         Product::query()
            ->whereNotIn('id', array(...$excludeIds, $product->{'id'}))
            ->where('business_id', $product->{'business_id'})
            ->where('business_category_id', $product->{'business_category_id'})
            ->where('creation_status', ProductStatusEnum::complete->value)
            ->whereNull('activated_at')
            ->where(function (Builder $builder) use ($product) {
               $builder
                  ->orWhere('business_sub_category_id', $product->{'business_sub_category_id'})
                  ->orWhere('business_sub_category_child_id', $product->{'business_sub_category_child_id'})
                  ->orWhere('business_sub_category_grand_child_id', $product->{'business_sub_category_grand_child_id'});
            })
            ->orderByDesc('created_at');


      if ($page === null) {
         $productData = $queryBuilder->get();
      } else {
         $paginator = $queryBuilder->paginate(
            perPage: config('torryme.constants.items_per_page'),
            page: $page,
         );
         $productData = $paginator->items();
      }

      // Goods
      if ($business->isGoodsBusiness()) {
         foreach ($productData as $product) {
            $products[] = $product->computeSampleAttributeForGoodsProduct(array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         }
      }

      // Services
      if ($business->isServiceBusiness()) {
         foreach ($productData as $product) {
            $products[] = $product->computeSampleAttributeForProductService(array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         }
      }

      // Cinema
      if ($business->isCinemaBusiness()) {
         foreach ($productData as $product) {
            $movie = $product->{'movie'};
            $series = $product->{'series'};
            $rating = $product->averageRating();

            if ($movie !== null) {
               $products[] = $product->computeSampleAttributeForProductMovie($movie, array(
                  "rating" => $rating,
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            } else if ($series !== null) {
               // Build generic data
               $products[] = $product->computeSampleAttributeForProductSeries($series, array(
                  "rating" => $rating,
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            }
         }
      }

      // Music
      if ($business->isMusicBusiness()) {
         foreach ($productData as $product) {
            $music = $product->{'music'};
            $musicAlbum = $product->{'musicAlbum'};

            if ($music !== null) {
               $products[] = $product->computeSampleAttributeForMusicProduct($music, array(
                  "rating" => $product->averageRating(),
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            } else if ($musicAlbum !== null) {
               $products[] = $product->computeSampleAttributeForAlbumMusicProduct($musicAlbum, array(
                  "rating" => $product->averageRating(),
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            }
         }
      }

      // Library
      if ($business->isLibraryBusiness()) {
         foreach ($productData as $product) {
            $book = $product->{'book'};

            if ($book !== null) {
               $products[] = $product->computeSampleAttributeForBookProduct($book, array(
                  "rating" => $product->averageRating(),
                  "liked" => $product->userLiked($user) ? 1 : 0,
                  "saved" => $product->userSaved($user) ? 1 : 0,
               ));
            }
         }
      }

      if ($page === null) {
         return $products;
      }

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         "products",
         $products,
      );
   }

   /**
    * Build query paginator for goods or service product
    *
    * @param Builder $source
    * @param int|null $page
    * @return LengthAwarePaginator
    */
   public static function goodsOrServiceProductQueryBuilder(Builder $source, int $page = null): LengthAwarePaginator
   {
      $queryBuilder =
         $source
            ->where(function (Builder $builder) {
               $builder->whereNull('game_id')
                  ->whereNull('music_id')
                  ->whereNull('music_album_id')
                  ->whereNull('book_id')
                  ->whereNull('movie_id')
                  ->whereNull('series_id');
            });

      return $queryBuilder->orderByDesc('created_at')->paginate(
         perPage: config('torryme.constants.items_per_page'),
         page: $page,
      );
   }

   // Utilities

   /**
    * Here, we calculate the most basics product field
    *
    * @param bool $withCartInfo
    * @return array
    */
   public function serializeCommonAttributes(bool $withCartInfo = true): array
   {
      $product = $this;
      $priceInfos = $product->convertPriceToAuthUserCurrency();

      $result = array(
         "id" => $product->{'id'},
         "designation" => $product->{'designation'},
         "description" => $product->{'description'},
         "price" => $priceInfos['price'],
         "price_currency" => $priceInfos['currency'],
         /*
         'price' => $product->{'price'},
         "price_currency" => $product->{'currency'}->{'code'},
         */
         "quantity" => $product->{'quantity'},
         "max_order" => $product->{'max_order'},
         "min_order" => $product->{'min_order'},
         "unit" => $product->{'unit'},
         "created_at" => Carbon::parse($product->{'created_at'})->utc(),
         "creation_status" => $product->{'creation_status'},
         "total_post" => 0,
         "hash_tags" => $product->getHashTagsAsString(),
         "business_category_code" => $product->{'businessCategory'}->{'code'},
         "owner_business" => $product->{'business'},
         "business_sub_category_designation" => $product->{'businessSubCategory'}->{'designation'},
         "business_sub_category_child_designation" => $product->{'businessSubCategoryChild'}->{'designation'},
         "business_sub_category_child_code" => $product->{'businessSubCategoryChild'}->{'code'},
         "business_sub_category_grand_child_designation" => $product->{'businessSubCategoryGrandChild'}?->{'designation'},
         "attachments" => $product->allAttachmentUrls(),
         "product_attributes" => ProductAttribute::computeForProductDetails($product),
         "already_purchased" => $product->checkProductAlreadyPurchased() ? 1 : 0
      );

      if ($withCartInfo) {
         /** @var ?User $user */
         $user = auth()->user();

         $result['cart_quantity'] = $user != null ? CartItem::getCartItemQuantity($user->{'id'}, $product->{'id'}) : null;
         $result['can_buy'] = $user != null ? ($this->userCanBuy($user) ? 1 : 0) : 0;
      }

      return $result;
   }

   /**
    * Serialize sample attributes
    *
    * @return array
    */
   public function serializeSampleAttributes(): array
   {
      $product = $this;

      return array(
         "id" => $product->{'id'},
         "designation" => $product->{'designation'},
         "business_sub_category_designation" => $product->{'businessSubCategory'}->{'designation'},
         "business_sub_category_child_designation" => $product->{'businessSubCategoryChild'}->{'designation'},
         "business_sub_category_grand_child_designation" => $product->{'businessSubCategoryGrandChild'}?->{'designation'},
      );
   }

   /**
    * Compute attributes for book product
    *
    * @param Book $book
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForBookProduct(Book $book, array $extraAttributes = []): array
   {
      $product = $this;
      $bookFileType = $book->bookFileType();

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,

         "book" => array(
            'id' => $book->{'id'},
            'cover_url' => $book->{'cover_url'},
            'book_url' => $book->{'book_url'},
            'size' => $book->bookSize(),
            'total_page' => $bookFileType === AttachmentTypeEnum::pdf->value ? $book->totalPage() : null,
            'book_type' => $bookFileType,
         )
      );
   }

   /**
    * Compute attributes for music album product
    *
    * @param MusicAlbum $musicAlbum
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForAlbumMusicProduct(MusicAlbum $musicAlbum, array $extraAttributes = []): array
   {
      $product = $this;

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,

         "album" => array(
            'id' => $musicAlbum->{'id'},
            'cover_url' => $musicAlbum->{'cover_url'},
            'size' => $musicAlbum->albumSize(),
            'total_single' => $musicAlbum->totalAlbumMusic(),
         )
      );
   }

   /**
    * Compute attributes for music product
    *
    * @param Music $music
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForMusicProduct(Music $music, array $extraAttributes = []): array
   {
      $product = $this;

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,

         "music" => array(
            'id' => $music->{'id'},
            'cover_url' => $music->{'cover_url'},
            'music_url' => $music->{'music_url'},
            'size' => $music->musicSize(),
            'duration' => $music->musicDuration(),
            'music_type' => $music->musicFileType(),
         )
      );
   }

   /**
    * Compute attributes for product service
    *
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForGoodsProduct(array $extraAttributes = []): array
   {
      $product = $this;

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,
      );
   }

   /**
    * Compute attributes for product service
    *
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForProductService(array $extraAttributes = []): array
   {
      $product = $this;

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,
         'delivery_in_days' => $product->{'delivery_in_days'}
      );
   }

   /**
    * Compute attributes for movie
    *
    * @param Movie $movie
    * @param array $extraAttributes
    * @return array
    */
   public function computeSampleAttributeForProductMovie(Movie $movie, array $extraAttributes = []): array
   {
      $product = $this;

      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,

         "movie" => array(
            'id' => $movie->{'id'},
            'cover_url' => $movie->{'cover_url'},
            'movie_url' => $movie->{'movie_url'},
            'size' => $movie->movieFileSize(),
            'duration' => $movie->movieDuration(),
         )
      );
   }

   /**
    * Compute attributes for series
    *
    * @param Series $series
    * @param array $extraAttributes
    * @param bool $withEpisodes
    * @return array
    */
   public function computeSampleAttributeForProductSeries(Series $series, array $extraAttributes = [], bool $withEpisodes = false): array
   {
      $product = $this;

      $seasons = SeriesSeason::computeSampleDetails($series, $series->{'seasons'}, $withEpisodes);

      // Build generic data
      return array(
         ...$product->serializeCommonAttributes(),
         ...$extraAttributes,

         "series" => array(
            'id' => $series->{'id'},
            'cover_url' => $series->{'cover_url'},
            'size' => $series->getTotalSize(),
            'total_episode' => $series->totalEpisode(),
            'total_seasons' => $series->totalSeriesSeason(),
            'seasons' => $seasons,
         )
      );
   }

   /**
    * Compute music shop for visitor
    *
    * @param $paginator
    * @return array
    */
   public static function computeMusicShopForVisitor($paginator): array
   {
      $products = [];
      $user = auth()->user();

      foreach ($paginator->items() as $product) {
         $music = $product->{'music'};
         $musicAlbum = $product->{'musicAlbum'};

         if ($music !== null) {
            $products[] = $product->computeSampleAttributeForMusicProduct($music, array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         } else if ($musicAlbum !== null) {
            $products[] = $product->computeSampleAttributeForAlbumMusicProduct($musicAlbum, array(
               "rating" => $product->averageRating(),
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         } else {
            // At this level, we don't have a music, we don't have
            // the album music, so the product is incomplete.
            $products[] = array(
               ...$product->serializeCommonAttributes(),
            );
         }
      }

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         "products",
         $products,
      );
   }

   /**
    * Compute cinema shop for visitor
    *
    * @param $paginator
    * @return array
    */
   public static function computeCinemaShopForVisitor($paginator): array
   {
      $products = [];
      $user = auth()->user();

      foreach ($paginator->items() as $product) {
         $movie = $product->{'movie'};
         $series = $product->{'series'};
         $rating = $product->averageRating();

         if ($movie !== null) {
            $products[] = $product->computeSampleAttributeForProductMovie($movie, array(
               "rating" => $rating,
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         } else if ($series !== null) {
            // Build generic data
            $products[] = $product->computeSampleAttributeForProductSeries($series, array(
               "rating" => $rating,
               "liked" => $product->userLiked($user) ? 1 : 0,
               "saved" => $product->userSaved($user) ? 1 : 0,
            ));
         } else {
            // At this level, we don't have a movie, we don't have
            // a series, so the product is incomplete.
            $products[] = array(
               ...$product->serializeCommonAttributes(),
            );
         }
      }

      return array(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         $products,
      );
   }
}
