<?php

namespace App\Services;

use App\Models\Level;
use App\Models\LevelClass;
use App\Models\SpecialtyClass;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait LevelService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Level::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all levels
     *
     * @return Builder[]|Collection
     */
    public static function allLevels(): Collection|array
    {
        return Level::query()->get();
    }

    /**
     * Create level
     *
     * @param string $designation
     * @param array $classIds
     * @return Level|Builder|Model|null
     */
    public static function createLevel(string $designation, array $classIds): Model|Builder|Level|null
    {
        DB::beginTransaction();

        try {
            /** @var Level $result */
            $level = self::store([
                "designation" => $designation,
            ]);

            if ($level != null) {
                $level->classes()->attach($classIds);
                $result = $level;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LevelService::createLevel');
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find level by id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return Level::query()->where('id', $id)->first();
    }

    /**
     * Update level with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $classIds
     * @return Builder|Model|null
     */
    public static function updateLevel(int $id, string $designation, array $classIds): Model|Builder|null
    {
        DB::beginTransaction();
        try {

            Level::query()->where('id', $id)->update(["designation" => $designation]);

            $level = self::findById($id);

            if ($level != null) {
                $level->classes()->sync($classIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LevelService::updateLevel');
            DB::rollBack();
            $level = null;
        }

        return $level;
    }

    /**
     * Delete level with given id
     *
     * @param int $id
     * @return Level|null
     */
    public static function deleteLevel(int $id): ?Level
    {
        DB::beginTransaction();
        try {

            /** @var Level $level */

            $level = self::findById($id);

            if ($level != null) {
                $level->classes()->detach();
                $level->delete();
                DB::commit();
            }
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LevelService::deleteLevel');
            DB::rollBack();
            $level = null;
        }

        return $level;
    }

    /**
     * Get Level class id
     *
     * @param int $levelId
     * @param int $classId
     * @return mixed
     */
    public static function getLevelClassId(int $levelId , int $classId): mixed
    {
        $levelClass = LevelClass::query()->where('level_id', $levelId)->where('class_id', $classId)->first();

        return $levelClass->{"id"};
    }

}
