<?php

namespace App\Services;

use App\Enums\LessonValidationEnum;
use App\Models\LessonProgression;
use App\Models\LessonProgressionDetail;
use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

trait LessonProgressionService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return LessonProgression::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all lesson progression
     *
     * @return Builder[]|Collection
     */
    public static function allLessonProgression(): Collection|array
    {
        return LessonProgression::query()->get();
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return LessonProgression::query()->With("lessonProgressionDetails")->find($id);
    }

    /**
     * Submit lesson for general education
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return LessonProgression|Builder|Model|null
     */
    public static function submitGeneralEducationLesson(array $data, string $educationTypeId, string $type): Model|Builder|LessonProgression|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $lessonProgression = null;

            /** @var User $user */
            $user = auth()->user();

            $path = self::handlePowerPointGeneralEducation($data);

            /** @var LessonProgression $results */

            foreach ($data['classIds'] as $classId) {
                $lessonProgression = self::store([
                    'inspectorate_id' => $data['inspectorate_id'],
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'class_id' => $classId,
                    'subject_id' => $data['subjectIds'][0],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'type' => $type,
                    'path' => $path,
                ]);
            }

            $lessonProgressionDetails = LessonProgressionDetail::submitGeneralEducationLessonDetails($data, $user, $lessonProgression->{"id"});
            $results = $lessonProgressionDetails;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::submitGeneralEducationLesson");
            self::deletePowerPoint($path);
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Submit technical education lesson
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return LessonProgression|Builder|Model|null
     */
    public static function submitTechnicalEducationLesson(array $data, string $educationTypeId, string $type): Model|Builder|LessonProgression|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $lessonProgression = null;

            /** @var User $user */
            $user = auth()->user();

            $path = self::handlePowerPointTechnicalEducation($data);

            /** @var LessonProgression $results */

            foreach ($data['specialtyClassIds'] as $classId) {
                $lessonProgression = self::store([
                    'inspectorate_id' => $data['inspectorate_id'],
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'sub_education_type_id' => $data['sub_education_type_id'],
                    'specialty_id' => $data['specialty_id'],
                    'class_id' => $classId,
                    'subject_id' => $data['subjectIds'][0],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'type' => $type,
                    'path' => $path,
                ]);
            }

            $lessonProgressionDetails = LessonProgressionDetail::submitTechnicalEducationLessonDetails($data, $user, $lessonProgression->{"id"});
            $results = $lessonProgressionDetails;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::submitTechnicalEducationLesson");
            self::deletePowerPoint($path);
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Submit teacher training lesson
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Builder|Model|null
     */
    public static function submitTeacherTrainingLesson(array $data, string $educationTypeId, string $type): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $lessonProgression = null;

            /** @var User $user */
            $user = auth()->user();

            $path = self::handlePowerPointTeacherTraining($data);

            foreach ($data['leveClassIds'] as $class) {
                $lessonProgression = self::store([
                    'inspectorate_id' => $data['inspectorate_id'],
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'level_id' => $data['level_id'],
                    'class_id' => $class,
                    'subject_id' => $data['subjectIds'][0],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'type' => $type,
                    'path' => $path,
                ]);
            }
            DB::commit();

            $lessonProgressionDetails = LessonProgressionDetail::submitTeacherTrainingLessonDetails($data, $user, $lessonProgression->{"id"});
            $results = $lessonProgressionDetails;
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::submitTeacherTrainingLesson");
            self::deletePowerPoint($path);
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Upload PowerPoint for general education
     *
     * @param array $data
     * @return mixed
     */
    public static function handlePowerPointGeneralEducation(array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $claas = classes($data['classIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_power_point'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload PowerPoint for technical education
     *
     * @param array $data
     * @return mixed
     */
    public static function handlePowerPointTechnicalEducation(array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $specialty = specialty($data['specialty_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $subEducationType = sub_education_type($data['sub_education_type_id']);
        $claas = classes($data['specialtyClassIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_power_point'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $subEducationType . '/' . $specialty . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload video for teacher training
     *
     * @param array $data
     * @return mixed
     */
    public static function handlePowerPointTeacherTraining(array $data): mixed
    {
        $level = level($data['level_id']);
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $claas = classes($data['leveClassIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_power_point'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $level . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Delete PowerPoint in given path
     *
     * @param string $path
     * @return bool
     */
    public static function deletePowerPoint(string $path): bool
    {
        return Storage::disk('public')->delete($path);
    }

    /**
     * Approve lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function approveLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $resultsArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetailGenEduc) {
                $resultsArray[] = array(
                    "id" => $lessonProgressionDetailGenEduc->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetailGenEduc->{"lesson_progression_id"},
                    "submitted_by" => $lessonProgressionDetailGenEduc->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetailGenEduc->{"submitted_at"},
                    "submitter_phone_number" => $lessonProgressionDetailGenEduc->{"submitter_phone_number"},
                    "validation_rejected_by" => $lessonProgressionDetailGenEduc->{"validation_rejected_by"},
                    "validation_rejected_at" => $lessonProgressionDetailGenEduc->{"validation_rejected_at"},
                    "validation_rejection_reason" => $lessonProgressionDetailGenEduc->{"validation_rejection_reason"},
                    "validated_by" => $lessonProgressionDetailGenEduc->{"validated_by"},
                    "validated_at" => $lessonProgressionDetailGenEduc->{"validated_at"},
                );
            }

            //Get last element in the array
            $lastLesson = end($resultsArray);

            //Get old PowerPoint path
            $lessonPath = $lessonProgression->{"path"};

            //Delete old powerpoint
            self::deletePowerPoint($lessonPath);

            if ($data["type"] == config('minesec.education_type.general')) {

                //Upload new PowerPoint to correct location
                $path = self::handleUpdatePowerPointGeneralEducation($lessonProgression, $data["lesson_power_point"]);

                //Update lesson path and status
                $lessonProgression->updateService(["path" => $path, "status" => LessonValidationEnum::validated->value]);
            }

            if ($data["type"] == config('minesec.education_type.technical')) {

                //Upload new PowerPoint to correct location
                $path = self::handleUpdatePowerPointTechnicalEducation($lessonProgression, $data["lesson_power_point"]);

                //Update lesson path and status
                $lessonProgression->updateService(["path" => $path, "status" => LessonValidationEnum::validated->value]);
            }

            if ($data["type"] == config('minesec.education_type.teacher_training')) {

                //Upload new PowerPoint to correct location
                $path = self::handleUpdatePowerPointTeacherTraining($lessonProgression, $data["lesson_power_point"]);

                //Update lesson path and status
                $lessonProgression->updateService(["path" => $path, "status" => LessonValidationEnum::validated->value]);
            }

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastLesson["lesson_progression_id"],
                "submitted_by" => $lastLesson["submitted_by"],
                "submitted_at" => date_format($lastLesson["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastLesson["submitter_phone_number"],
                'validated_by' => $user->{"id"},
                'validated_at' => now(),
            ]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::approveLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Reject lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function rejectLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $lastValueArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetail) {
                $lastValueArray[] = array(
                    "id" => $lessonProgressionDetail->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetail->{"lesson_progression_id"},
                    "submitted_by" => $lessonProgressionDetail->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetail->{"submitted_at"},
                    "submitter_phone_number" => $lessonProgressionDetail->{"submitter_phone_number"},
                    "validated_by" => $lessonProgressionDetail->{"validated_by"},
                    "validated_at" => $lessonProgressionDetail->{"validated_at"},
                    "validation_rejected_by" => $lessonProgressionDetail->{"validation_rejected_by"},
                    "validation_rejected_at" => $lessonProgressionDetail->{"validation_rejected_at"},
                    "validation_rejection_reason" => $lessonProgressionDetail->{"validation_rejection_reason"},
                );
            }

            $lastValue = end($lastValueArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastValue["lesson_progression_id"],
                "submitted_by" => $lastValue["submitted_by"],
                "submitted_at" => date_format($lastValue["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastValue["submitter_phone_number"],
                'validated_by' => null,
                'validated_at' => null,
                'validation_rejected_by' => $user->{"id"},
                'validation_rejected_at' => now(),
                'validation_rejection_reason' => $data["reason"],
            ]);

            //Update lesson status
            $lessonProgression->updateService(["status" => LessonValidationEnum::rejected->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::rejectLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Update PowerPoint for General Education
     *
     * @param $lessonProgression
     * @param $powerPoint
     * @return mixed
     */
    public static function handleUpdatePowerPointGeneralEducation($lessonProgression, $powerPoint): mixed
    {
        $subSystem = sub_system($lessonProgression->{'sub_system_id'});
        $educationType = education_type($lessonProgression->{'education_type_id'});
        $claas = classes($lessonProgression->{'class_id'});
        $subject = subject($lessonProgression->{'subject_id'});

        $file = $powerPoint;
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Update PowerPoint for Technical Education
     *
     * @param $lessonProgression
     * @param $powerPoint
     * @return mixed
     */
    public static function handleUpdatePowerPointTechnicalEducation($lessonProgression, $powerPoint): mixed
    {
        $subSystem = sub_system($lessonProgression->{'sub_system_id'});
        $educationType = education_type($lessonProgression->{'education_type_id'});
        $subEducationType = sub_education_type($lessonProgression->{'sub_education_type_id'});
        $specialty = specialty($lessonProgression->{'specialty_id'});
        $claas = classes($lessonProgression->{'class_id'});
        $subject = subject($lessonProgression->{'subject_id'});

        $file = $powerPoint;
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $subEducationType . '/' . $specialty . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Update PowerPoint for Teach Training
     *
     * @param $lessonProgression
     * @param $powerPoint
     * @return mixed
     */
    public static function handleUpdatePowerPointTeacherTraining($lessonProgression, $powerPoint): mixed
    {
        $subSystem = sub_system($lessonProgression->{'sub_system_id'});
        $educationType = education_type($lessonProgression->{'education_type_id'});
        $level = level($lessonProgression->{'level_id'});
        $claas = classes($lessonProgression->{'class_id'});
        $subject = subject($lessonProgression->{'subject_id'});

        $file = $powerPoint;
        $name = $file->getClientOriginalName();

        return $file->storeAs('PowerPoints/' . $subSystem . '/' . $educationType . '/' . $level . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Get all validated lessons
     *
     * @return Builder[]|Collection
     */
    public static function validatedLessons(): Collection|array
    {
        $status = LessonValidationEnum::validated->value;
        $rejectedRecordings = LessonValidationEnum::recordingRejected->value;

        return LessonProgression::query()->where("status", $status)->orWhere("status", $rejectedRecordings)->get();
    }

    /**
     * Approve validates lesson
     *
     * @param int $id
     * @return Builder|Model|null
     */
    public static function approveValidatedLesson(int $id): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($id);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($id);

            $resultsArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetailGenEduc) {
                $resultsArray[] = array(
                    "id" => $lessonProgressionDetailGenEduc->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetailGenEduc->{"lesson_progression_id"},
                    "submitted_at" => $lessonProgressionDetailGenEduc->{"submitted_at"},
                    "submitter_phone_number" => $lessonProgressionDetailGenEduc->{"submitter_phone_number"},
                    "submitted_by" => $lessonProgressionDetailGenEduc->{"submitted_by"},
                );
            }

            //Get last element in the array
            $lastLesson = end($resultsArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastLesson["lesson_progression_id"],
                "submitted_by" => $lastLesson["submitted_by"],
                "submitted_at" => date_format($lastLesson["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastLesson["submitter_phone_number"],
                "recorded_by" => $user->{"id"},
                "recorded_at" => now(),
            ]);

            $lessonProgression->updateService(["status" => LessonValidationEnum::recorded->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::approveValidatedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Reject validated lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function rejectValidatedLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {

            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $lastValueArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetail) {
                $lastValueArray[] = array(
                    "id" => $lessonProgressionDetail->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetail->{"lesson_progression_id"},
                    "submitted_by" => $lessonProgressionDetail->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetail->{"submitted_at"},
                    "submitter_phone_number" => $lessonProgressionDetail->{"submitter_phone_number"},
                );
            }

            $lastValue = end($lastValueArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastValue["lesson_progression_id"],
                "submitted_by" => $lastValue["submitted_by"],
                "submitted_at" => date_format($lastValue["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastValue["submitter_phone_number"],
                'recording_rejected_by' => $user->{"id"},
                'recording_rejected_at' => now(),
                'recording_rejection_reason' => $data["reason"],
            ]);

            //Update lesson status
            $lessonProgression->updateService(["status" => LessonValidationEnum::ValidationRejected->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::rejectValidatedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Get all validated lessons
     *
     * @return Builder[]|Collection
     */
    public static function recordedLessons(): Collection|array
    {
        $status = LessonValidationEnum::recorded->value;

        return LessonProgression::query()->where("status", $status)->get();
    }

    /**
     * Approve recorded lesson
     *
     * @param int $id
     * @return Builder|Model|null
     */
    public static function approveRecordedLesson(int $id): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($id);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($id);

            $resultsArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetailGenEduc) {
                $resultsArray[] = array(
                    "id" => $lessonProgressionDetailGenEduc->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetailGenEduc->{"lesson_progression_id"},
                    "submitted_by" => $lessonProgressionDetailGenEduc->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetailGenEduc->{"submitted_at"},
                    "submitter_phone_number" => $lessonProgressionDetailGenEduc->{"submitter_phone_number"},
                );
            }

            //Get last element in the array
            $lastLesson = end($resultsArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastLesson["lesson_progression_id"],
                "submitted_by" => $lastLesson["submitted_by"],
                "submitted_at" => date_format($lastLesson["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastLesson["submitter_phone_number"],
                "edited_by" => $user->{"id"},
                "edited_at" => now(),
            ]);


            $lessonProgression->updateService(["status" => LessonValidationEnum::edited->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::approveRecordedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Reject recorded lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function rejectRecordedLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();

        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $lastValueArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetail) {
                $lastValueArray[] = array(
                    "id" => $lessonProgressionDetail->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetail->{"lesson_progression_id"},
                    "submitter_phone_number" => $lessonProgressionDetail->{"submitter_phone_number"},
                    "submitted_by" => $lessonProgressionDetail->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetail->{"submitted_at"},
                );
            }

            $lastValue = end($lastValueArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastValue["lesson_progression_id"],
                "submitted_by" => $lastValue["submitted_by"],
                "submitted_at" => date_format($lastValue["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastValue["submitter_phone_number"],
                "editing_rejected_by" => $user->{"id"},
                "editing_rejected_at" => now(),
                "editing_rejection_reason" => $data["reason"],
            ]);

            //Update lesson status
            $lessonProgression->updateService(["status" => LessonValidationEnum::recordingRejected->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::rejectRecordedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Get edited lesson
     *
     * @return Builder[]|Collection
     */
    public static function editedLessons(): Collection|array
    {
        $status = LessonValidationEnum::edited->value;

        return LessonProgression::query()->where("status", $status)->get();
    }

    /**
     * Approve edited lesson
     *
     * @param int $id
     * @return Builder|Model|null
     */
    public static function approveEditedLesson(int $id): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($id);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($id);

            $resultsArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetailGenEduc) {
                $resultsArray[] = array(
                    "id" => $lessonProgressionDetailGenEduc->{"id"},
                    "lesson_progression_id" => $lessonProgressionDetailGenEduc->{"lesson_progression_id"},
                    "submitter_phone_number" => $lessonProgressionDetailGenEduc->{"submitter_phone_number"},
                    "submitted_by" => $lessonProgressionDetailGenEduc->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetailGenEduc->{"submitted_at"},
                );
            }

            //Get last element in the array
            $lastLesson = end($resultsArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastLesson["lesson_progression_id"],
                "submitted_by" => $lastLesson["submitted_by"],
                "submitted_at" => date_format($lastLesson["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastLesson["submitter_phone_number"],
                "segmented_by" => $user->{"id"},
                "segmented_at" => now(),
            ]);

            $lessonProgression->updateService(["status" => LessonValidationEnum::segmented->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::approveEditedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Reject edited lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function rejectedEditedLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();
        try {

            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $lastValueArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetail) {
                $lastValueArray[] = array(
                    "id" => $lessonProgressionDetail->{"id"},
                    "submitter_phone_number" => $lessonProgressionDetail->{"submitter_phone_number"},
                    "lesson_progression_id" => $lessonProgressionDetail->{"lesson_progression_id"},
                    "submitted_by" => $lessonProgressionDetail->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetail->{"submitted_at"},
                );
            }

            $lastValue = end($lastValueArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastValue["lesson_progression_id"],
                "submitted_by" => $lastValue["submitted_by"],
                "submitted_at" => date_format($lastValue["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastValue["submitter_phone_number"],
                "segmenting_rejected_by" => $user->{"id"},
                "segmenting_rejected_at" => now(),
                "segmenting_rejection_reason" => $data["reason"],
            ]);

            //Update lesson status
            $lessonProgression->updateService(["status" => LessonValidationEnum::editingRejected->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::rejectedEditedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Get segmented lessons
     *
     * @return Builder[]|Collection
     */
    public static function segmentedLessons(): Collection|array
    {
        $status = LessonValidationEnum::segmented->value;

        return LessonProgression::query()->where("status", $status)->get();
    }

    /**
     * Put segmented lesson on class
     *
     * @param int $id
     * @return Builder|Model|null
     */
    public static function putSegmentedLessonInClass(int $id): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($id);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($id);

            $resultsArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetailGenEduc) {
                $resultsArray[] = array(
                    "id" => $lessonProgressionDetailGenEduc->{"id"},
                    "submitter_phone_number" => $lessonProgressionDetailGenEduc->{"submitter_phone_number"},
                    "submitted_by" => $lessonProgressionDetailGenEduc->{"submitted_by"},
                    "submitted_at" => $lessonProgressionDetailGenEduc->{"submitted_at"},
                    "lesson_progression_id" => $lessonProgressionDetailGenEduc->{"lesson_progression_id"},
                );
            }

            //Get last element in the array
            $lastLesson = end($resultsArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastLesson["lesson_progression_id"],
                "submitted_by" => $lastLesson["submitted_by"],
                "submitted_at" => date_format($lastLesson["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastLesson["submitter_phone_number"],
                "put_in_classes_by" => $user->{"id"},
                "put_in_classes_at" => now(),
            ]);

            $lessonProgression->updateService(["status" => LessonValidationEnum::putInClass->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::putSegmentedLessonInClass");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Reject segmented lesson
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function rejectSegmentedLesson(array $data): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            /** @var User $user */
            $user = auth()->user();

            $lessonProgression = LessonProgression::findById($data["lesson_progression_id"]);

            $lessonProgressionDetails = LessonProgressionDetail::findByLessonProgressionId($data["lesson_progression_id"]);

            $lastValueArray = array();
            foreach ($lessonProgressionDetails as $lessonProgressionDetail) {
                $lastValueArray[] = array(
                    "id" => $lessonProgressionDetail->{"id"},
                    "submitted_by" => $lessonProgressionDetail->{"submitted_by"},
                    "submitter_phone_number" => $lessonProgressionDetail->{"submitter_phone_number"},
                    "lesson_progression_id" => $lessonProgressionDetail->{"lesson_progression_id"},
                    "submitted_at" => $lessonProgressionDetail->{"submitted_at"},
                );
            }

            $lastValue = end($lastValueArray);

            $newLessonProgressionDetail = LessonProgressionDetail::store([
                "lesson_progression_id" => $lastValue["lesson_progression_id"],
                "submitted_by" => $lastValue["submitted_by"],
                "submitted_at" => date_format($lastValue["submitted_at"], "Y-m-d h:i:s"),
                "submitter_phone_number" => $lastValue["submitter_phone_number"],
                "put_in_classes_rejected_by" => $user->{"id"},
                "put_in_classes_rejected_at" => now(),
                "put_in_classes_rejection_reason" => $data["reason"],
            ]);

            //Update lesson status
            $lessonProgression->updateService(["status" => LessonValidationEnum::segmentationRejected->value]);

            $results = $newLessonProgressionDetail;

            DB::commit();
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionService::rejectSegmentedLesson");
            DB::rollBack();
            $results = null;
        }

        return $results;
    }
}
