<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\Specialty;
use App\Models\SubEducationType;
use App\RequestRules\Web\SpecialtyRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class SpecialtyController extends Controller
{

    /**
     * Get the specialty list view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {

        $classes = Classes::allClasses();
        $specialties = Specialty::allSpecialties();

        return view("admin.specialty.index", compact("classes", "specialties"));
    }

    /**
     * Create specialty
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(SpecialtyRules::create());

        $designation = $request->{"designation"};
        $classIds = $request->{"classIds"};

        $response = Specialty::createSpecialty($designation, $classIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('specialty.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get specialty with given id
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public  function edit(Request $request): View|Factory|Application
    {
        $request->validate(SpecialtyRules::edit($request->merge(["id" => $request->route("id")])));
        $id = $request->{"id"};

        $specialty= Specialty::findById($id);
        $classes = Classes::allClasses();
        $specialties = Specialty::allSpecialties();

        return view("admin.specialty.edit", compact("specialty", "classes", "specialties"));

    }

    /**
     * Update specialty with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(SpecialtyRules::update());

        $id = $request->{"id"};
        $designation = $request->{"designation"};
        $classIds = $request->{"classIds"};

        $response = Specialty::updateSpecialty($id,$designation, $classIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('specialty.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete specialty with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(SpecialtyRules::delete());
        $id = $request->{"id"};

        $response = Specialty::deleteSpecialty($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get specialties
     *
     * @param Request $request
     * @return JsonResponse
     */
    public  function  getSpecialties(Request $request): JsonResponse
    {
        $request->validate(SpecialtyRules::getSpecialties());

        $id = $request->{"sub_education_type_id"};

        $subEducationType = SubEducationType::findById($id);

        $specialties = array();
        foreach ($subEducationType->{"specialties"} as $specialty) {
            $specialties[] =array(
                "id" => $specialty->{"id"},
                "designation" => $specialty->{"designation"}
            );
        }

        return response()->json($specialties);
    }
}
