<?php

namespace App\Services;

use App\Models\EducationType;
use App\Models\Level;
use App\Models\Specialty;
use App\Models\SubjectGroup;
use App\Models\SubjectGroupClass;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait SubjectGroupService
{

    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SubjectGroup::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }


    /**
     * Get all subjects
     *
     * @return Builder[]|Collection
     */
    public static function allSubjectGroups(): Collection|array
    {
        return SubjectGroup::query()->get();
    }

    /**
     * Find by given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return SubjectGroup::query()->find($id);
    }

    /**
     * Create class subjects
     *
     * @param array $data
     * @param int $educationTypeId
     * @param string $type
     * @return Model|Builder|SubjectGroup|null
     */
    public static function createSubjectGroup(array $data, int $educationTypeId, string $type): Model|Builder|SubjectGroup|null
    {
        DB::beginTransaction();

        try {
            /** @var SubjectGroup $result */

            $subjectGroup = self::store([
                "designation" => $data["designation"],
            ]);

            if ($type != null && $type == config('minesec.education_type.general')) {

                $educationTypeClassId = EducationType::getEducationTypeClassId($educationTypeId, $data["class_id"]);

                $subjectGroupId = $subjectGroup->{"id"};

                SubjectGroupClass::createSubjectGroupGeneralEducation($subjectGroupId, $data["sub_system_id"], $educationTypeId, $educationTypeClassId);

                $subjectGroup?->subjectGroupSubjects()->attach($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

            if ($type != null && $type == config('minesec.education_type.technical')) {

                $specialtyClassId = Specialty::getSpecialtyClassId($data["specialty_id"], $data["specialty_class_id"]);

                $subjectGroupId = $subjectGroup->{"id"};

                SubjectGroupClass::createSubjectGroupTechnicalEducation($subjectGroupId, $data["sub_system_id"], $educationTypeId, $data["sub_education_type_id"], $specialtyClassId);

                $subjectGroup?->subjectGroupSubjects()->attach($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

            if ($type != null && $type == config('minesec.education_type.teacher_training')) {

                $levelClassId = Level::getLevelClassId($data["level_id"], $data["level_class_id"]);

                $subjectGroupId = $subjectGroup->{"id"};

                SubjectGroupClass::createSubjectGroupTeacherTraining($subjectGroupId, $data["sub_system_id"], $educationTypeId, $levelClassId);

                $subjectGroup?->subjectGroupSubjects()->attach($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubjectGroupService::createSubjectGroup');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Edit sub system subject
     *
     * @param array $data
     * @param int $educationTypeId
     * @param string $type
     * @return SubjectGroup|null
     */
    public static function updateSubjectGroup(array $data, int $educationTypeId, string $type): ?SubjectGroup
    {
        DB::beginTransaction();

        try {
            /** @var SubjectGroup $result */

            /** @var SubjectGroup $subjectGroup */
            $subjectGroup = SubjectGroup::findById($data["subject_group_id"]);
            $subjectGroup->updateService([
                "designation" => $data["designation"],
            ]);

            if ($type != null && $type == config('minesec.education_type.general')) {
                $educationTypeClassId = EducationType::getEducationTypeClassId($educationTypeId, $data["class_id"]);

                /** @var SubjectGroupClass $subjectGroupClass */
                $subjectGroupClass = SubjectGroupClass::findBySubjectGroupId($data["subject_group_id"]);
                $subjectGroupClass->update([
                    "sub_system_id" => $data["sub_system"],
                    "education_type_id" => $educationTypeId,
                    "education_type_class_id" => $educationTypeClassId
                ]);

                $subjectGroup->subjectGroupSubjects()->sync($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

            if ($type != null && $type == config('minesec.education_type.technical')) {
                $specialtyClassId = Specialty::getSpecialtyClassId($data["specialty"], $data["specialty_class"]);

                /** @var SubjectGroupClass $subjectGroupClass */
                $subjectGroupClass = SubjectGroupClass::findBySubjectGroupId($data["subject_group_id"]);
                $subjectGroupClass->update([
                    "sub_system_id" => $data["sub_system"],
                    "education_type_id" => $educationTypeId,
                    "sub_education_type_id" => $data["sub_education_type"],
                    "specialty_class_id" => $specialtyClassId
                ]);

                $subjectGroup->subjectGroupSubjects()->sync($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

            if ($type != null && $type == config('minesec.education_type.teacher_training')) {
                $levelClassId = Level::getLevelClassId($data["level"], $data["level_class"]);

                /** @var SubjectGroupClass $subjectGroupClass */
                $subjectGroupClass = SubjectGroupClass::findBySubjectGroupId($data["subject_group_id"]);
                $subjectGroupClass->update([
                    "sub_system_id" => $data["sub_system"],
                    "education_type_id" => $educationTypeId,
                    "level_class_id" => $levelClassId
                ]);

                $subjectGroup->subjectGroupSubjects()->sync($data["subjectIds"]);

                $result = $subjectGroup;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubjectGroupService::updateSubjectGroup');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Delete subject group with given id
     *
     * @param int $id
     * @return SubjectGroup|null
     */
    public static function deleteClassSubject(int $id): ?SubjectGroup
    {
        DB::beginTransaction();

        try {
            /** @var SubjectGroup $cLassSubject */

            $cLassSubject = self::findById($id);
            $subjectGroupClass = SubjectGroupClass::findBySubjectGroupId($id);

            if($cLassSubject != null && $subjectGroupClass != null) {
                $cLassSubject->subjectGroupSubjects()->detach();
                $cLassSubject->delete();
                $subjectGroupClass->forceDelete();
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubjectGroupService::deleteClassSubject');
            DB::rollBack();
            $cLassSubject = null;
        }

        return $cLassSubject;
    }

    /**
     * Get class subject
     *
     * @param string $type
     * @param array $data
     * @return Builder|Model|null
     */
    public static  function getClassSubjects(string $type, array $data): Model|Builder|null
    {
        if($type == config('minesec.education_type.general')) {

            $educationTypeClassId = EducationType::getEducationTypeClassId($data["education_type_id"], $data["class_id"]);

            return SubjectGroupClass::getClassSubjectGroup($type, $educationTypeClassId);
        }

        if($type == config('minesec.education_type.technical')) {

            $specialtyClassId = Specialty::getSpecialtyClassId($data["specialty_id"], $data["specialty_class_id"]);

            return SubjectGroupClass::getClassSubjectGroup($type, $specialtyClassId);
        }

        if($type == config('minesec.education_type.teacher_training')) {

            $levelClassId = Level::getLevelClassId($data["level_id"], $data["level_class_id"]);

            return SubjectGroupClass::getClassSubjectGroup($type, $levelClassId);
        }

        return  null;
    }

}
