<?php

namespace App\Services;

use App\Models\Role;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait RoleService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Role::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all roles
     *
     * @return Builder[]|Collection
     */
    public static function allRoles(): Collection|array
    {
        return Role::query()->get();
    }

    /**
     * Find by role id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return Role::query()->find($id);
    }

    /**
     * Get all roles
     *
     * @return Builder[]|Collection
     */
    public static function getSystemRoles(): Collection|array
    {
        return Role::query()->where("is_system", config("minesec.role_types.system_roles"))->get();
    }

    /**
     * Get inspectorate roles
     *
     * @return Builder[]|Collection
     */
    public static function getInspectorateRoles(): Collection|array
    {
        return Role::query()->where("is_system", config("minesec.role_types.non_system_roles"))->get();
    }

    /**
     * @param array $data
     * @return \Spatie\Permission\Models\Role|\Spatie\Permission\Contracts\Role|null
     */
    public static function assignPermissionsToRole(array $data): \Spatie\Permission\Models\Role|null|\Spatie\Permission\Contracts\Role
    {
        try {
            $role = \Spatie\Permission\Models\Role::findById($data["role_id"]);

            $assignedPermissions = $role->syncPermissions($data["permissionIds"]);
            $result = $assignedPermissions;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RoleService::assignPermissionsToRole');
            $result = null;
        }

        return $result;
    }

}
