<?php

namespace App\Services;

use App\Models\Region;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait RegionService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Region::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all regions
     *
     * @return Builder[]|Collection
     */
    public static function allRegions(): Collection|array
    {
        return Region::query()->get();
    }

    /**
     * create a new region
     *
     * @param string $designation
     * @param array $divisionIds
     * @return Region|Builder|Model|null
     */
    public static function createRegion(string $designation, array $divisionIds): Model|Builder|Region|null
    {
        DB::beginTransaction();

        try {
            /** @var Region $result */
            $region = self::store([
                "designation" => $designation,
            ]);

            if($region != null) {
                $region->divisions()->attach($divisionIds);
                $result = $region;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RegionService::createRegion');
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find region by id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return Region::query()->find($id);
    }

    /**
     * Update region with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $divisionIds
     * @return Model|Builder|null
     */
    public static function updateRegion(int $id, string $designation, array $divisionIds): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            Region::query()->where("id", $id)->update(["designation" => $designation]);

            $region = self::findById($id);

            if($region != null) {
                $region->divisions()->sync($divisionIds);
                DB::commit();
            }
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RegionService::updateRegion');
            DB::rollBack();
            $region = null;
        }

        return $region;
    }

    /**
     * Delete region with given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function deleteRegion(int $id): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            $region = self::findById($id);

            if($region != null) {
                $region->divisions()->detach();
                $region->delete();
                DB::commit();
            }
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RegionService::deleteRegion');
            DB::rollBack();
            $region = null;
        }

        return $region;
    }
}
