<?php

namespace App\Services;

use App\Models\EducationType;
use App\Models\EducationTypeClass;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait EducationTypeService
{

    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return EducationType::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all Education types
     *
     * @return Builder[]|Collection
     */
    public static function allEducationTypes(): Collection|array
    {
        return EducationType::query()->get();
    }

    /**
     * Create education type - technical
     *
     * @param string $designation
     * @param array $subEducationTypeIds
     * @param string $type
     * @return EducationType|Builder|Model|null
     */
    public static function createTechnicalEducation(string $designation, array $subEducationTypeIds, string $type): Model|Builder|EducationType|null
    {
        DB::beginTransaction();
        try {
            /** @var EducationType $result */
            $educationType = self::store([
                "designation" => $designation,
                "type" => $type,
            ]);

            if ($educationType != null) {
                $educationType->subEducation()->attach($subEducationTypeIds);
                $result = $educationType;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::createTechnicalEducation");
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Create education type - general
     *
     * @param string $designation
     * @param array $classIds
     * @param string $type
     * @return EducationType|Builder|Model|null
     */
    public static function createGeneralEducation(string $designation, array $classIds, string $type): Model|Builder|EducationType|null
    {
        DB::beginTransaction();
        try {
            /** @var EducationType $result */
            $educationType = self::store([
                "designation" => $designation,
                "type" => $type,
            ]);

            if ($educationType != null) {
                $educationType->classes()->attach($classIds);
                $result = $educationType;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::createGeneralEducation");
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Create education type - teacher training
     *
     * @param string $designation
     * @param array $levelIds
     * @param string $type
     * @return EducationType|Builder|Model|null
     */
    public static function createTeacherTraining(string $designation, array $levelIds, string $type): Model|Builder|EducationType|null
    {
        DB::beginTransaction();
        try {

            /** @var EducationType $result */
            $educationType = self::store([
                "designation" => $designation,
                "type" => $type,
            ]);

            if ($educationType != null) {
                $educationType->Levels()->attach($levelIds);
                $result = $educationType;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::createTeacherTraining");
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find Education Type by id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return EducationType::query()->where("id", $id)->first();
    }

    /**
     * Update technical education type with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $subEducationTypeIds
     * @param string $type
     * @return Model|Builder|null
     */
    public static function updateTechnicalEducation(int $id, string $designation, array $subEducationTypeIds, string $type): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            EducationType::query()->where('id', $id)->update(["designation" => $designation, "type" => $type]);

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->subEducation()->sync($subEducationTypeIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::updateTechnicalEducation");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }


    /**
     * Update general education type with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $classIds
     * @param string $type
     * @return Model|Builder|null
     */
    public static function updateGeneralEducation(int $id, string $designation, array $classIds, string $type): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            EducationType::query()->where('id', $id)->update(["designation" => $designation, "type" => $type]);

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->classes()->sync($classIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::updateGeneralEducation");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }

    /**
     * Update teacher training with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $levelIds
     * @param string $type
     * @return Builder|Model|null
     */
    public static function updateTeacherTrainingEducation(int $id, string $designation, array $levelIds, string $type): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            EducationType::query()->where('id', $id)->update(["designation" => $designation, "type" => $type]);

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->levels()->sync($levelIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::updateTeacherTrainingEducation");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }

    /**
     * Delete technical education type with given id
     *
     * @param int $id
     * @return EducationType|null
     */
    public static function deleteTechnicalEducation(int $id): ?EducationType
    {
        DB::beginTransaction();
        try {
            /** @var EducationType $educationType */

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->subEducation()->detach();
                $educationType->delete();
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::deleteTechnicalEducation");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }


    /**
     * Delete general education type with given id
     *
     * @param int $id
     * @return EducationType|null
     */
    public static function deleteGeneralEducation(int $id): ?EducationType
    {
        DB::beginTransaction();
        try {
            /** @var EducationType $educationType */

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->classes()->detach();
                $educationType->delete();
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::deleteGeneralEducation");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }

    /**
     * Delete teacher traing with given id
     *
     * @param int $id
     * @return EducationType|null
     */
    public static function deleteTeacherTraining(int $id): ?EducationType
    {
        DB::beginTransaction();
        try {
            /** @var EducationType $educationType */

            $educationType = self::findById($id);

            if ($educationType != null) {
                $educationType->levels()->detach();
                $educationType->delete();
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug($exception, "EducationTypeService::deleteTeacherTraining");
            DB::rollBack();
            $educationType = null;
        }

        return $educationType;
    }

    /**
     * Get education type
     *
     * @param int $id
     * @return mixed
     */
    public static function getEducationType(int $id): mixed
    {
        $type = EducationType::findById($id);

        return $type->{"type"};
    }

    /**
     * Get Education type class id
     *
     * @param int $educationTypeId
     * @param int $classId
     * @return mixed
     */
    public static function getEducationTypeClassId(int $educationTypeId, int $classId): mixed
    {
        $educationType = EducationTypeClass::query()->where("education_type_id", $educationTypeId)->where("class_id", $classId)->first();

        return $educationType->{"id"};
    }
}
