<?php

namespace App\Services;

use App\Models\Division;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait DivisionService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Division::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all divisions
     *
     * @return Builder[]|Collection
     */
    public static function allDivisions(): Collection|array
    {
        return Division::query()->orderBy('designation')->get();
    }

    /**
     * create division
     *
     * @param string $designation
     * @return Division|Builder|Model|null
     * @throws Exception
     */
    public static function createDivision(string $designation): Division|Model|Builder|null
    {
        try {
            /** @var Division $result */
            $division = self::store([
                "designation" => $designation,
            ]);

            $result = $division;

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'DivisionService::createDivision');
            $result = null;
        }

        return $result;
    }

    /**
     * Find division by id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return Division::query()->find($id);
    }

    /**
     * Update division with the given id
     *
     * @param int $divisionId
     * @param string $designation
     * @return int|null
     */
    public static function updateDivision(int $divisionId, string $designation): ?int
    {
        try {
            $division = Division::query()->where("id", $divisionId)->update(["designation" => $designation]);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'DivisionService::updateDivision');
            $division = null;
        }
        return $division;
    }

    /** Delete division with given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function deleteDivision(int $id): Model|Builder|null
    {
        try {
            $division = self::findById($id);

            $division?->delete();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'DivisionService::deleteDivision');
            $division = null;
        }

        return $division;
    }
}
