<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\Term;
use App\RequestRules\Web\TermRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class TermController extends Controller
{

    /**
     * Get term list view
     *
     * @return Application|Factory|View
     */
    public  function index(): View|Factory|Application
    {

        $terms = Term::allTerms();

        return view("admin.term.index", compact("terms"));
    }

    /**
     * Create term
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {

        $request->validate(TermRules::create());

        $designation = $request->{"designation"};

        $response = Term::createTerm($designation);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('term.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Return edit view
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(TermRules::edit($request->merge(["id" => $request->route("id")])));

        $id = $request->{"id"};

        $term = Term::findById($id);
        $terms = Term::allTerms();

        return view("admin.term.edit", compact("term", "terms"));
    }

    /**
     * Update term with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(TermRules::update());

        $id = $request->{"id"};
        $designation = $request->{"designation"};

        $response = Term::updateTerm($id, $designation);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('term.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete term with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(TermRules::delete());

        $id = $request->{"id"};

        $response = Term::deleteTerm($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }
}
