<?php

namespace App\Services;

use App\Models\Lesson;
use App\Models\LessonProgression;
use App\Models\LessonTitle;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait LessonTitleService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return LessonTitle::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get lesson titles
     *
     * @return Builder[]|Collection
     */
    public static function getLessonTitles(): Collection|array
    {
        return LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])->get();
    }

    /**
     * Create general education lesson titles
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Builder|Model|null
     */
    public static function createGeneralEducation(array $data, int $educationTypeId): Model|Builder|null
    {
        try {
            $results = null;

            foreach ($data["lesson_titles"] as $key => $value) {
                $results = self::store([
                    "title" => $value,
                    "subject_group_id" => $data["subject_group_id"],
                    "sub_system_id" => $data["subject_group_sub_system_id"],
                    "education_type_id" => $educationTypeId,
                    "class_id" => $data["class_id"],
                    "subject_id" => $data["subject_id"],
                ]);
            }
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonTitleService::createGeneralEducation');
            $results = null;
        }

        return $results;
    }

    /**
     * Create technical education lesson titles
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Builder|Model|null
     */
    public static function createTechnicalEducation(array $data, int $educationTypeId): Model|Builder|null
    {
        try {
            $results = null;

            foreach ($data["lesson_titles"] as $key => $value) {
                $results = self::store([
                    "title" => $value,
                    "subject_group_id" => $data["subject_group_id"],
                    "sub_system_id" => $data["subject_group_sub_system_id"],
                    "education_type_id" => $educationTypeId,
                    "sub_education_type_id" => $data["sub_education_type_id"],
                    "specialty_id" => $data["specialty_id"],
                    "class_id" => $data["specialty_class_id"],
                    "subject_id" => $data["subject_id"],
                ]);
            }

        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonTitleService::createTechnicalEducation');
            $results = null;
        }

        return $results;
    }

    /**
     * Create teacher training lesson tittles
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Builder|Model|null
     */
    public static function createTeacherTrainingEducation(array $data, int $educationTypeId): Model|Builder|null
    {
        try {
            $results = null;

            foreach ($data["lesson_titles"] as $key => $value) {
                $results = self::store([
                    "title" => $value,
                    "subject_group_id" => $data["subject_group_id"],
                    "sub_system_id" => $data["subject_group_sub_system_id"],
                    "education_type_id" => $educationTypeId,
                    "level_id" => $data["level_id"],
                    "class_id" => $data["level_class_id"],
                    "subject_id" => $data["subject_id"],
                ]);
            }

        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonTitleService::createTeacherTrainingEducation');
            $results = null;
        }

        return $results;
    }

    /**
     * Get General Education subject lesson titles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $classId
     * @param int $subjectId
     * @param string|null $requestType
     * @return Builder[]|Collection
     */
    public static function getGeneralEducationSubjectLessonTitles(int $subSystemId, int $educationTypeId, int $classId, int $subjectId, string $requestType = null): Collection|array
    {
        $lessonTitlesArray = [];

        if ($requestType != null & $requestType == "lessons to digitised") {
            $lessonTitles = Lesson::getGenEducLessonTitles($subSystemId, $educationTypeId, $classId, $subjectId);

            $digitisedLessonsIds = array();
            foreach ($lessonTitles as $LessonTitles) {
                $digitisedLessonsIds[] = $LessonTitles->{"lesson_title_id"};
            }

          $lessonsToDigitised =  LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
                ->where("sub_system_id", $subSystemId)
                ->where("education_type_id", $educationTypeId)
                ->where("class_id", $classId)
                ->where("subject_id", $subjectId)
                ->get();

            $lessonArray = array();
            foreach ($lessonsToDigitised as $lessonToDigitised) {
                if(array_search($lessonToDigitised->{"id"}, $digitisedLessonsIds) == false) {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        "description" => $lessonToDigitised->{"description"},
                        "status" => __("messages.not_digitised"),
                        'key' => "not_yet",
                    );
                } else {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        "description" => $lessonToDigitised->{"description"},
                        "status" => __("messages.digitised"),
                        'key' => "digitised",
                    );
                }
            }

            return $lessonArray;
        }

        if ($requestType != null & $requestType == "lesson progression") {
            $lessonTitles = LessonProgression::getGenEducLessonProgressionLessonTitles($subSystemId, $educationTypeId, $classId, $subjectId);

            $lessonTitlesArray = array();
            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        if ($requestType != null & $requestType == "class lessons") {
            $lessonTitles = Lesson::getGenEducLessonTitles($subSystemId, $educationTypeId, $classId, $subjectId);

            $lessonTitlesArray = array();
            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        return LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
            ->where("sub_system_id", $subSystemId)
            ->where("education_type_id", $educationTypeId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)
            ->whereNotIn('id', $lessonTitlesArray)
            ->get();
    }

    /**
     * Get Technical Education subject lesson titles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $subEducationTypeId
     * @param int $specialtyId
     * @param int $classId
     * @param int $subjectId
     * @param string|null $requestType
     * @return Collection|array
     */
    public static function getTechnicalEducationSubjectLessonTitles(int $subSystemId, int $educationTypeId, int $subEducationTypeId, int $specialtyId, int $classId, int $subjectId, string $requestType = null): Collection|array
    {
        $lessonTitlesArray = array();

        if ($requestType != null & $requestType == "lessons to digitised") {
            $lessonTitles = Lesson::getTechEducLessonTitles($subSystemId, $educationTypeId, $subEducationTypeId, $specialtyId, $classId, $subjectId);

            $digitisedLessonsIds = array();
            foreach ($lessonTitles as $LessonTitles) {
                $digitisedLessonsIds[] = $LessonTitles->{"lesson_title_id"};
            }

            $lessonsToDigitised = LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
                ->where("sub_system_id", $subSystemId)
                ->where("education_type_id", $educationTypeId)
                ->where("sub_education_type_id", $subEducationTypeId)
                ->where("specialty_id", $specialtyId)
                ->where("class_id", $classId)
                ->where("subject_id", $subjectId)
                ->get();

            $lessonArray = array();
            foreach ($lessonsToDigitised as $lessonToDigitised) {
                if(array_search($lessonToDigitised->{"id"}, $digitisedLessonsIds) == false) {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        "description" => $lessonToDigitised->{"description"},
                        'key' => "not_yet",
                        "status" => __("messages.not_digitised"),
                    );
                } else {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        "description" => $lessonToDigitised->{"description"},
                        'key' => "digitised",
                        "status" => __("messages.digitised"),
                    );
                }
            }

            return $lessonArray;
        }

        if ($requestType != null & $requestType == "lesson progression") {
            $lessonTitles = LessonProgression::getTechEducLessonProgressionLessonTitles($subSystemId, $educationTypeId, $subEducationTypeId, $specialtyId, $classId, $subjectId);

            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        if ($requestType != null & $requestType == "class lessons") {
            $lessonTitles = Lesson::getTechEducLessonTitles($subSystemId, $educationTypeId, $subEducationTypeId, $specialtyId, $classId, $subjectId);

            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        return LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
            ->where("sub_system_id", $subSystemId)
            ->where("education_type_id", $educationTypeId)
            ->where("sub_education_type_id", $subEducationTypeId)
            ->where("specialty_id", $specialtyId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)
            ->whereNotIn('id', $lessonTitlesArray)
            ->get();
    }

    /**
     * Get Technical Education subject lesson titles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $levelId
     * @param int $classId
     * @param int $subjectId
     * @param string|null $requestType
     * @return Builder[]|Collection
     */
    public static function getTeacherTrainingSubjectLessonTitles(int $subSystemId, int $educationTypeId, int $levelId, int $classId, int $subjectId, string $requestType = null): Collection|array
    {
        $lessonTitlesArray = null;

        if ($requestType != null & $requestType == "lessons to digitised") {
            $lessonTitles = Lesson::getTechTrainLessonTitles($subSystemId, $educationTypeId, $levelId, $classId, $subjectId);

            $digitisedLessonsIds = array();
            foreach ($lessonTitles as $LessonTitles) {
                $digitisedLessonsIds[] = $LessonTitles->{"lesson_title_id"};
            }

            $lessonsToDigitised = LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
                ->where("sub_system_id", $subSystemId)
                ->where("education_type_id", $educationTypeId)
                ->where("level_id", $levelId)
                ->where("class_id", $classId)
                ->where("subject_id", $subjectId)
                ->get();

            $lessonArray = array();
            foreach ($lessonsToDigitised as $lessonToDigitised) {
                if(array_search($lessonToDigitised->{"id"}, $digitisedLessonsIds) == false) {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        'key' => "not_yet",
                        "status" => __("messages.not_digitised"),
                        "description" => $lessonToDigitised->{"description"},
                    );
                } else {
                    $lessonArray[] = array(
                        "id" => $lessonToDigitised->{"id"},
                        "title" => $lessonToDigitised->{"title"},
                        'key' => "digitised",
                        "status" => __("messages.digitised"),
                        "description" => $lessonToDigitised->{"description"},
                    );
                }
            }

            return $lessonArray;
        }

        if ($requestType != null & $requestType == "lesson progression") {
            $lessonTitles = LessonProgression::getTechTrainLessonProgressionLessonTitles($subSystemId, $educationTypeId, $levelId, $classId, $subjectId);
            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        if ($requestType != null & $requestType == "class lessons") {
            $lessonTitles = Lesson::getTechTrainLessonTitles($subSystemId, $educationTypeId, $levelId, $classId, $subjectId);
            foreach ($lessonTitles as $LessonTitles) {
                $lessonTitlesArray[] = $LessonTitles->{"lesson_title_id"};
            }
        }

        return LessonTitle::query()->with(["subSystem", "educationType", "subEducationType", "classes", "specialty", "level"])
            ->where("sub_system_id", $subSystemId)
            ->where("education_type_id", $educationTypeId)
            ->where("level_id", $levelId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)
            ->whereNotIn('id', $lessonTitlesArray)
            ->get();
    }

}
