<?php

namespace App\Services;

use App\Enums\ResourceUploadTypeEnum;
use App\Models\Lesson;
use App\Models\Subject;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use ProtoneMedia\LaravelFFMpeg\Support\FFMpeg;

trait LessonService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Lesson::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all lessons
     *
     * @return Builder[]|Collection
     */
    public static function allLessons(): Collection|array
    {
        return Lesson::query()->with(['lessonTitle','subSystem', 'educationType', 'className', 'subject', 'term'])->get();
    }

    /**
     * Find lesson with given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return Lesson::query()->where('id', $id)->first();
    }

    /**
     * Create lesson for general education
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonGeneralEducation(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $thumbnailPath = null;

            if ($data["upload_type"] == ResourceUploadTypeEnum::localServer->value) {
                $path = self::handleVideoGeneralEducation($data);
                $thumbnailPath = self::handleThumbnailGeneration($data, $type);
                $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();

                if ($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour')) {
                    $duration = gmdate("H:i:s", $lessonLengthInSeconds);
                } else {
                    $duration = gmdate("i:s", $lessonLengthInSeconds);
                }
            } else {
                $path = $data["youtube_lesson_link"];
                $thumbnailPath = $data["youtube_lesson_link"];
                $duration = $data["youtube_lesson_link"];
            }

            //Get subject inspectorate
            $subjectInspectorateId = Subject::getSubjectInspectorateId($data['subject_id']);

            /** @var Lesson $results */
            $lesson = self::store([
                'lesson_title_id' => $data['lesson_title_id'],
                'sub_system_id' => $data['sub_system_id'],
                'education_type_id' => $educationTypeId,
                'class_id' => $data['class_id'],
                'inspectorate_id' => $subjectInspectorateId,
                'subject_id' => $data['subject_id'],
                'term_id' => $data['term_id'],
                'upload_type' => $data["upload_type"],
                'duration' => $duration,
                'type' => $type,
                'path' => $path,
                'thumbnail' => $thumbnailPath,
            ]);

            DB::commit();
            $results = $lesson;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonGeneralEducation');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create lesson for technical education
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonTechnicalEducation(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $thumbnailPath = null;

            if ($data["upload_type"] == ResourceUploadTypeEnum::localServer->value) {
                $path = self::handleVideTechnicalEducation($data);
                $thumbnailPath = self::handleThumbnailGeneration($data, $type);
                $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();

                if ($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour')) {
                    $duration = gmdate("H:i:s", $lessonLengthInSeconds);
                } else {
                    $duration = gmdate("i:s", $lessonLengthInSeconds);
                }
            } else {
                $path = $data["youtube_lesson_link"];
                $thumbnailPath = $data["youtube_lesson_link"];
                $duration = $data["youtube_lesson_link"];
            }

            //Get subject inspectorate
            $subjectInspectorateId = Subject::getSubjectInspectorateId($data['subject_id']);

            /** @var Lesson $results */;
            $lesson = self::store([
                'lesson_title_id' => $data['lesson_title_id'],
                'sub_system_id' => $data['sub_system_id'],
                'education_type_id' => $educationTypeId,
                'sub_education_type_id' => $data['sub_education_type_id'],
                'specialty_id' => $data['specialty_id'],
                'class_id' => $data['specialty_class_id'],
                'inspectorate_id' => $subjectInspectorateId,
                'subject_id' => $data['subject_id'],
                'term_id' => $data['term_id'],
                'upload_type' => $data["upload_type"],
                'duration' => $duration,
                'type' => $type,
                'path' => $path,
                'thumbnail' => $thumbnailPath,
            ]);

//            ConvertLessonForStreaming::dispatch($results);
            DB::commit();

            $results = $lesson;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonTechnicalEducation');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create lesson for teacher training
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonTeacherTraining(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $thumbnailPath = null;
            $path = null;

            if ($data["upload_type"] == ResourceUploadTypeEnum::localServer->value) {
                $path = self::handleVideTeacherTraining($data);
                $thumbnailPath = self::handleThumbnailGeneration($data, $type);
                $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();
                if ($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour')) {
                    $duration = gmdate("H:i:s", $lessonLengthInSeconds);
                } else {
                    $duration = gmdate("i:s", $lessonLengthInSeconds);
                }
            } else {
                $path = $data["youtube_lesson_link"];
                $thumbnailPath = $data["youtube_lesson_link"];
                $duration = $data["youtube_lesson_link"];
            }

            //Get subject inspectorate
            $subjectInspectorateId = Subject::getSubjectInspectorateId($data['subject_id']);

            /** @var Lesson $results */
            $lesson = self::store([
                'lesson_title_id' => $data['lesson_title_id'],
                'sub_system_id' => $data['sub_system_id'],
                'education_type_id' => $educationTypeId,
                'level_id' => $data['level_id'],
                'class_id' => $data['level_class_id'],
                'inspectorate_id' => $subjectInspectorateId,
                'subject_id' => $data['subject_id'],
                'term_id' => $data['term_id'],
                'upload_type' => $data["upload_type"],
                'duration' => $duration,
                'type' => $type,
                'path' => $path,
                'thumbnail' => $thumbnailPath,
            ]);

            DB::commit();
            $results = $lesson;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonTeacherTraining');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * @param $filename
     * @return string
     */
    private static function getCleanFileName($filename): string
    {
        return str_replace('.mp4', '', str_replace(' ', '-', strtolower($filename))) . '.png';
    }

    /**s
     * Handle thumbnail capture and  uploading
     *
     * @param array $data
     * @param string $type
     * @return string|null
     */
    public static function handleThumbnailGeneration(array $data, string $type): ?string
    {
        if ($type == config('minesec.education_type.general')) {

            $subSystem = sub_system($data['sub_system_id']);
            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $claas = classes($data['class_id']);
            $subject = subject($data['subject_id']);

            $thumbNailName = self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $claas . '/' . $subject . '/' . $thumbNailName);

            return 'lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $claas . '/' . $subject . '/' . $thumbNailName;
        }

        if ($type == config('minesec.education_type.technical')) {

            $subject = subject($data['subject_id']);
            $subSystem = sub_system($data['sub_system_id']);
            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $subEducationType = sub_education_type($data['sub_education_type_id']);
            $specialty = specialty($data['specialty_id']);
            $claas = classes($data['specialty_class_id']);

            $thumbNailName = self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $subEducationType . '/' . $specialty . '/' . $claas . '/' . $subject . '/' . $thumbNailName);

            return 'lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $subEducationType . '/' . $specialty . '/' . $claas . '/' . $subject . '/' . $thumbNailName;
        }

        if ($type == config('minesec.education_type.teacher_training')) {

            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $level = level($data['level_id']);
            $claas = classes($data['level_class_id']);
            $subject = subject($data['subject_id']);
            $subSystem = sub_system($data['sub_system_id']);

            $thumbNailName = self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $level . '/' . $claas . '/' . $subject . '/' . 'FrameAt10sec.png');

            return 'lesson_thumbnails/' . $subSystem . '/' . $educationType . '/' . $level . '/' . $claas . '/' . $subject . '/' . $thumbNailName;
        }

        return null;
    }

    /**
     * Upload video for general education
     *
     * @param array $data
     * @return mixed
     */
    public static function handleVideoGeneralEducation(array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $claas = classes($data['class_id']);
        $subject = subject($data['subject_id']);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/' . $subSystem . '/' . $educationType . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload video for technical education
     *
     * @param array $data
     * @return mixed
     */
    public static function handleVideTechnicalEducation(array $data): mixed
    {
        $specialty = specialty($data['specialty_id']);
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $subEducationType = sub_education_type($data['sub_education_type_id']);
        $claas = classes($data['specialty_class_id']);
        $subject = subject($data['subject_id']);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/' . $subSystem . '/' . $educationType . '/' . $subEducationType . '/' . $specialty . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload video for teacher training
     *
     * @param array $data
     * @return mixed
     */
    public static function handleVideTeacherTraining(array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $level = level($data['level_id']);
        $claas = classes($data['level_class_id']);
        $subject = subject($data['subject_id']);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/' . $subSystem . '/' . $educationType . '/' . $level . '/' . $claas . '/' . $subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Delete lesson with given id
     *
     * @param int $id
     * @return Lesson|null
     */
    public static function deleteLesson(int $id): ?Lesson
    {
        DB::beginTransaction();
        try {
            /** @var Lesson $results */

            $lesson = self::findById($id);
            $path = $lesson->{"path"};
            $thumbnail = $lesson->{"thumbnail"};
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnail);
            self::destroy($id);
            $results = $lesson;
            DB::commit();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::deleteLesson');
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Delete video in given path
     *
     * @param string $path
     * @return bool
     */
    public static function deleteVideo(string $path): bool
    {
        return Storage::disk('public')->delete($path);

    }

    /**
     * @param string $thumbNailPath
     * @return bool
     */
    public static function deleteThumbnail(string $thumbNailPath): bool
    {
        return Storage::disk('public')->delete($thumbNailPath);
    }

    /**
     * Get subject lessons
     *
     * @param array $data
     * @return Builder[]|Collection
     */
    public static function subjectLessons(array $data): Collection|array
    {
        $lessons = Lesson::query()->with('lessonTitle');

        if ($data["type"] == config('minesec.education_type.general')) {
            $lessons->whereNull(array('sub_education_type_id', 'specialty_id', 'level_id'))
                ->where('type', $data["type"])
                ->where('sub_system_id', $data["sub_system_id"])
                ->where('education_type_id', $data["education_type_id"])
                ->where('class_id', $data["class_id"])
                ->where('subject_id', $data["subject_id"]);
        }

        if ($data["type"] == config('minesec.education_type.technical')) {
            $lessons->whereNull('level_id')
                ->where('type', $data["type"])
                ->where('sub_system_id', $data["sub_system_id"])
                ->where('education_type_id', $data["education_type_id"])
                ->where('sub_education_type_id', $data["sub_education_type_id"])
                ->where('specialty_id', $data["specialty_id"])
                ->where('class_id', $data["class_id"])
                ->where('subject_id', $data["subject_id"]);
        }

        if ($data["type"] == config('minesec.education_type.teacher_training')) {
            $lessons->whereNull(array('sub_education_type_id', 'specialty_id'))
                ->where('type', $data["type"])
                ->where('sub_system_id', $data["sub_system_id"])
                ->where('education_type_id', $data["education_type_id"])
                ->where('level_id', $data["level_id"])
                ->where('class_id', $data["class_id"])
                ->where('subject_id', $data["subject_id"]);
        }

        return $lessons->get();
    }

    /**
     * Get lesson statistics
     *
     * @return array
     */
    public static function lessonStatistics(): array
    {
        return array(
            "english_sub_system" => Lesson::query()->where("sub_system_id", config("minesec.sub_systems.english_sub_system"))->count(),
            "general_education" => Lesson::query()->where("education_type_id", config("minesec.education_types.general_education"))->count(),
            "technical_education" => Lesson::query()->where("education_type_id", config("minesec.education_types.technical_education"))->count(),
            "teacher_training" => Lesson::query()->where("education_type_id", config("minesec.education_types.teacher_training"))->count(),

            "sous_systeme_francophone" => Lesson::query()->where("sub_system_id", config("minesec.sub_systems.sou_system_francophone"))->count(),
            "enseignement_general" => Lesson::query()->where("education_type_id", config("minesec.education_types.enseignement_general"))->count(),
            "enseignement_technique" => Lesson::query()->where("education_type_id", config("minesec.education_types.enseignement_technique"))->count(),
            "enseignement_Normal" => Lesson::query()->where("education_type_id", config("minesec.education_types.enseignement_normal"))->count(),

            "total_lessons" => Lesson::query()->count(),
            "total_general_education" => Lesson::query()->where("type", config('minesec.education_type.general'))->count(),
            "total_technical_education" => Lesson::query()->where("type", config('minesec.education_type.technical'))->count(),
            "total_teacher_training" => Lesson::query()->where("type", config('minesec.education_type.teacher_training'))->count(),
        );

    }

    /**
     * Get number of lessons by inspectorate
     *
     * @param int $inspectorateId
     * @return int
     */
    public static function inspectorateStatistics(int $inspectorateId): int
    {
        return Lesson::query()->where("inspectorate_id", $inspectorateId)->count();
    }

    /**
     * Get General Education lesson tittles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $classId
     * @param int $subjectId
     * @return Builder[]|Collection
     */
    public static function getGenEducLessonTitles(int $subSystemId, int $educationTypeId, int $classId, int $subjectId): array|Collection
    {
        return Lesson::query()->where('sub_system_id', $subSystemId)
            ->where('education_type_id' , $educationTypeId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)->get();
    }

    /**
     * Get Technical Education lesson titles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $subEducationTypeId
     * @param int $specialtyId
     * @param int $classId
     * @param int $subjectId
     * @return array|Collection
     */
    public static function getTechEducLessonTitles(int $subSystemId, int $educationTypeId, int $subEducationTypeId, int $specialtyId, int $classId, int $subjectId): array|Collection
    {
        return Lesson::query()->where('sub_system_id', $subSystemId)
            ->where('education_type_id' , $educationTypeId)
            ->where('sub_education_type_id' , $subEducationTypeId)
            ->where('specialty_id' , $specialtyId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)->get();
    }

    /**
     * Get Teacher Training lesson titles
     *
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $levelId
     * @param int $classId
     * @param int $subjectId
     * @return array|Collection
     */
    public static function getTechTrainLessonTitles(int $subSystemId, int $educationTypeId, int $levelId, int $classId, int $subjectId): array|Collection
    {
        return Lesson::query()->where('sub_system_id', $subSystemId)
            ->where('education_type_id' , $educationTypeId)
            ->where('level_id' , $levelId)
            ->where("class_id", $classId)
            ->where("subject_id", $subjectId)->get();
    }


}
