<?php

namespace App\Http\Controllers\Web\Inspectorate;

use App\Http\Controllers\Controller;
use App\Libraries\ConstantsLib;
use App\Models\DigitisationPlanning;
use App\Models\EducationType;
use App\Models\Region;
use App\Models\School;
use App\Models\Session;
use App\Models\SubSystem;
use App\RequestRules\Web\DigitisationRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class DigitisationController extends Controller
{

    /**
     * @var ConstantsLib
     */
    protected ConstantsLib $ConstantsLib;

    public function __construct(ConstantsLib $ConstantsLib)
    {
        $this->ConstantsLib = $ConstantsLib;
    }

    /**
     * Get index view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function index(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $subSystems = SubSystem::allSubSystems();
        $schools = School::allSchools();
        $sessions = Session::allSessions();
        $regions = Region::allRegions();
        $digitisationStages = $this->ConstantsLib->getDigitisationStage();

        return view("inspectorate.digitalisation.index", compact("subSystems", "schools", "sessions", "regions", "digitisationStages"));
    }

    /**
     * Get divisions
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getDivisions(Request $request): JsonResponse
    {
        $request->validate(DigitisationRules::getDivisions());
        $regionId = $request->{"region_id"};

        $region = Region::findById($regionId);
        $divisions = array();
        foreach ($region->{"divisions"} as $division) {
            $divisions[] = array(
                "id" => $division->{"id"},
                "designation" => $division->{"designation"}
            );
        }

        return response()->json($divisions);
    }

    /**
     * Create Digitisation planning
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function planning(Request $request): JsonResponse
    {
        $data = $request->all();
        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(DigitisationRules::addGenEducPlanning());
            $response = DigitisationPlanning::createGenEducationPlanning($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(DigitisationRules::AddTechEducPlanning());
            $response = DigitisationPlanning::createTechEducationPlanning($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(DigitisationRules::addTeachTrainingPlanning());
            $response = DigitisationPlanning::createTeachTrainingPlanning($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        return response()->json(["error" => __("error.generic_error")]);
    }


    public function getPlannedLessons(Request $request)
    {
        $data = $request->all();
        $lessonTitles = null;

        if ($data["type"] == config('minesec.education_type.general')) {
            $request->validate(DigitisationRules::getGenEducPlannedLessons());
            $lessonTitles = DigitisationPlanning::getGenEducationPlanning($data);
        }

        if ($data["type"] == config('minesec.education_type.technical')) {
            $request->validate(DigitisationRules::getTechEducPlanning());
            $lessonTitles = DigitisationPlanning::getTechEducationPlanning($data);
        }

        if ($data["type"] == config('minesec.education_type.teacher_training')) {
            $request->validate(DigitisationRules::getTeachTrainingPlanning());
            $lessonTitles = DigitisationPlanning::getTeachTrainingPlanning($data);
        }

        if ($lessonTitles === null) {
            return api_response(150, 'No lessons found');
        }

        return api_response(100, 'ok', $lessonTitles);
    }

    public function implementation()
    {

    }

}
