<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Imports\InspectorateSubjectImport;
use App\Models\EducationType;
use App\Models\Inspectorate;
use App\Models\Subject;
use App\Models\SubjectGroup;
use App\Models\SubSystem;
use App\RequestRules\Web\SubjectRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Redirect;
use Maatwebsite\Excel\Facades\Excel;

class SubjectController extends Controller
{
    /**
     * Get subject list view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {

        $subjects = Subject::allSubjects();
        $inspectorates = Inspectorate::allInspectorates();

        return view("admin.subject.index", compact("subjects", "inspectorates"));
    }

    /**
     * Create subject
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(SubjectRules::create());
        $data = $request->all();

        if(isset($data["upload_file"])) {
            $request->session()->now('inspectorate_id', $data["inspectorate_id"]);
            Excel::import(new InspectorateSubjectImport, $request->file('subjects'));
            return Redirect::route('subject.index')->with("success", __("messages.record_successfully_imported"));
        }

        $response = Subject::createSubject($data);
        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('subject.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get subject with given id
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(SubjectRules::edit($request->merge(["id" => $request->route("id")])));

        $id = $request->{"id"};

        $subject = Subject::findById($id);
        $subjects = Subject::allSubjects();
        $inspectorates = Inspectorate::allInspectorates();

        return view("admin.subject.edit", compact("subject", "subjects", "inspectorates"));
    }

    /**
     * Update subject with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(SubjectRules::update());
        $data = $request->all();

        $response = Subject::updateSubject($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('subject.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete subject with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::delete());

        $id = $request->{"id"};

        $response = Subject::deleteSubject($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get sub system - subject view
     *
     * @return Factory|View|Application
     */
    public function subSystemSubjects(): Factory|View|Application
    {
        $subSystems = SubSystem::allSubSystems();

        $subjects = Subject::allSubjects();

        return view("admin.subject.sub_system.index", compact("subSystems", "subjects"));
    }

    /**
     * Create sub system subject
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function createSubSystemSubjects(Request $request): RedirectResponse
    {
        $request->validate(SubjectRules::createOrUpdateSubSystemSubjects());

        $subSystemId = $request->{"sub_system_id"};
        $subjectIds = $request->{"subjectIds"};

        $response = SubSystem::createSubSystemSubjects($subSystemId, $subjectIds);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('subject.subSystem-subjects')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get edit sub system subject view
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function editSubSystemSubject(Request $request): View|Factory|Application
    {
        $request->validate(SubjectRules::editSubSystemSubjects($request->merge(["id" => $request->route("id")])));

        $id = $request->{"id"};

        $currentSubsystem = SubSystem::findById($id);
        $subSystems = SubSystem::allSubSystems();
        $subjects = Subject::allSubjects();

        return view("admin.subject.sub_system.edit", compact("currentSubsystem", "subSystems", "subjects"));
    }

    /**
     * Update sub system subject with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function updateSubSystemSubject(Request $request): RedirectResponse
    {
        $request->validate(SubjectRules::createOrUpdateSubSystemSubjects());

        $subSystemId = $request->{"sub_system_id"};
        $subjectIds = $request->{"subjectIds"};

        $response = SubSystem::updateSubSystemSubjects($subSystemId, $subjectIds);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('subject.subSystem-subjects')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete sub system subjects with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteSubSystemSubject(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::deleteSubSystemSubjects());;

        $id = $request->{"id"};

        $response = SubSystem::deleteSubSystemSubjects($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get sub system subjects
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getSubSystemSubjects(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::getSubSystemSubjects());

        $id = $request->{"sub_system_id"};

        $subSystem = SubSystem::findById($id);

        $subjectResult = array();
        foreach ($subSystem->{"subjects"} as $subject) {
            $subjectResult[] = array(
                "id" => $subject->{"id"},
                "designation" => $subject->{"designation"}
            );
        }

        return response()->json($subjectResult);
    }

    /**
     * Get selected sub system subjects
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getSelectedSubSystemSubjects(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::getSelectedSubSystemSubjects());
        $data = $request->all();

        $subSystem = SubSystem::findById($data["sub_system_id"]);
        $subjectResult = array();
        foreach ($subSystem->{"subjects"} as $subject) {
            $subjectResult[] = array(
                "id" => $subject->{"id"},
                "designation" => $subject->{"designation"}
            );
        }

        $subjectGroup = SubjectGroup::findById($data["subject_group_id"]);
        $selectedSubjectsArray = array();
        foreach ($subjectGroup->{"subjectGroupSubjects"} as $selectedSubjects) {
            $selectedSubjectsArray[] = array(
                "id" => $selectedSubjects->{"id"},
                "designation" => $selectedSubjects->{"designation"}
            );
        }

        return response()->json(["subjects" => $subjectResult, "selectedSubjects" => $selectedSubjectsArray]);
    }

    /**
     * Get class subject view
     *
     * @return View|Factory|Application
     */
    public function ClassSubjects(): View|Factory|Application
    {
        $subSystems = SubSystem::allSubSystems();
        $subjectGroups = SubjectGroup::allSubjectGroups();

        return view("admin.subject.class.index", compact("subSystems", "subjectGroups"));
    }

    /**
     * Create class subject
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function createClassSubject(Request $request): RedirectResponse
    {
        $data = $request->all();

        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(SubjectRules::createClassSubjectGeneralEducation());

            $response = SubjectGroup::createSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_created"));
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(SubjectRules::createClassSubjectTechnicalEducation());

            $response = SubjectGroup::createSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_created"));
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(SubjectRules::createClassSubjectTeacherTraining());

            $response = SubjectGroup::createSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_created"));
        }

        return Redirect::route('subject.class-subject')->with("error", __("error.generic_error"));
    }

    /**
     * Get edit sub system subject view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function editClassSubject(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $request->validate(SubjectRules::editClassSubjects($request->merge(["id" => $request->route("id")])));

        $subSystems = SubSystem::allSubSystems();
        $subjectGroups = SubjectGroup::allSubjectGroups();
        $subjectGroup = SubjectGroup::findById($request->{"id"});

        return view("admin.subject.class.edit", compact("subSystems", "subjectGroups", "subjectGroup"));
    }

    /**
     * Update Sub system subject
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function updateClassSubject(Request $request): RedirectResponse
    {
        $data = $request->all();

        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(SubjectRules::updateClassSubjectGeneralEducation());

            $response = SubjectGroup::updateSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_updated"));
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(SubjectRules::updateClassSubjectTechnicalEducation());

            $response = SubjectGroup::updateSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_updated"));
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(SubjectRules::updateClassSubjectTeacherTraining());

            $response = SubjectGroup::updateSubjectGroup($data, $educationTypeId, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.class-subject')->with("success", __("messages.record_successfully_updated"));
        }


        return Redirect::route('subject.class-subject')->with("error", __("error.generic_error"));;
    }

    /**
     * Delete subject with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteCLassSubject(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::deleteClassSubject());

        $id = $request->{"id"};

        $response = SubjectGroup::deleteClassSubject($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get class subjects
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function getClassSubjects(Request $request): Application|ResponseFactory|Response
    {
        $data = $request->all();
        $subjectGroupClass = null;

        if ($data["type"] == config('minesec.education_type.general')) {
            $request->validate(SubjectRules::getGeneralEducationClassSubjects());

            $subjectGroupClass = SubjectGroup::getClassSubjects($data["type"], $data);
        }

        if ($data["type"] == config('minesec.education_type.teacher_training')) {
            $request->validate(SubjectRules::getTeacherTrainingClassSubjects());

            $subjectGroupClass = SubjectGroup::getClassSubjects($data["type"], $data);
        }

        if ($data["type"] == config('minesec.education_type.technical')) {
            $request->validate(SubjectRules::getTechnicalEducationClassSubjects());

            $subjectGroupClass = SubjectGroup::getClassSubjects($data["type"], $data);
        }

        if ($subjectGroupClass === null) {
            return api_response(150, 'No subjects found');
        }

        $subjectGroup = SubjectGroup::findById($subjectGroupClass->{"subject_group_id"});
        $subjectGroupResults = array();

        foreach ($subjectGroup->{"subjectGroupSubjects"} as $subject) {
            $subjectGroupResults[] = array(
                "id" => $subject->{"id"},
                "designation" => $subject->{"designation"}
            );
        }

        return api_response(100, 'ok', $subjectGroupResults);
    }

    /**
     * get subject group class subject
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getSubjectGroupClassSubject(Request $request): JsonResponse
    {
        $request->validate(SubjectRules::getSubjectGroupSubjects());
        $subjectGroup = SubjectGroup::findById($request->{"subject_group_id"});

        $subjectGroupCLassSubjects = $subjectGroup->{"subjectGroupSubjects"};

        $subjectGroupCLassSubjectArray = array();
        foreach ($subjectGroupCLassSubjects as $subject) {
            $subjectGroupCLassSubjectArray[] = array(
                "id" => $subject->{"id"},
                "designation" => $subject->{"designation"},
            );
        }

        return response()->json($subjectGroupCLassSubjectArray);
    }

}
