<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Imports\GeneralEducationLessonTitleImport;
use App\Imports\TeacherTrainingLessonTitleImport;
use App\Imports\TechnicalEducationLessonTitleImport;
use App\Models\EducationType;
use App\Models\LessonTitle;
use App\Models\SubjectGroup;
use App\RequestRules\Web\LessonTitleRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Redirect;
use Maatwebsite\Excel\Facades\Excel;

class LessonTitleController extends Controller
{
    /**
     * Lesson title view
     *
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function index(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $subjectGroups = SubjectGroup::allSubjectGroups();
        $lessonTitles = LessonTitle::getLessonTitles();

        return view("admin.subject.lesson_titles.index", compact("subjectGroups", "lessonTitles"));
    }

    /**
     * Create general education lesson titles
     *
     * @param Request $request
     * @return JsonResponse|RedirectResponse
     */
    public function create(Request $request): JsonResponse|RedirectResponse
    {
        $data = $request->all();

        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(LessonTitleRules::createGeneralEducation());

            if (isset($data["upload_file"])) {
                $request->session()->now('subject_group_id', $data["subject_group_id"]);
                $request->session()->now('sub_system_id', $data["subject_group_sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('class_id', $data["class_id"]);
                $request->session()->now('subject_id', $data["subject_id"]);

                Excel::import(new GeneralEducationLessonTitleImport, $request->file('lesson_titles'));
                return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_imported"));
            }

            $response = LessonTitle::createGeneralEducation($data, $educationTypeId);
            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_created"));
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(LessonTitleRules::createTechnicalEducation());

            if (isset($data["upload_file"])) {
                $request->session()->now('subject_group_id', $data["subject_group_id"]);
                $request->session()->now('sub_system_id', $data["subject_group_sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('sub_education_type_id', $data["sub_education_type_id"]);
                $request->session()->now('specialty_id', $data["specialty_id"]);
                $request->session()->now('class_id', $data["specialty_class_id"]);
                $request->session()->now('subject_id', $data["subject_id"]);

                Excel::import(new TechnicalEducationLessonTitleImport, $request->file('lesson_titles'));
                return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_imported"));
            }

            $response = LessonTitle::createTechnicalEducation($data, $educationTypeId);
            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_created"));
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(LessonTitleRules::createTeacherTrainingEducation());
            if (isset($data["upload_file"])) {
                $request->session()->now('subject_group_id', $data["subject_group_id"]);
                $request->session()->now('sub_system_id', $data["subject_group_sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('level_id', $data["level_id"]);
                $request->session()->now('class_id', $data["level_class_id"]);
                $request->session()->now('subject_id', $data["subject_id"]);

                Excel::import(new TeacherTrainingLessonTitleImport, $request->file('lesson_titles'));
                return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_imported"));
            }

            $response = LessonTitle::createTeacherTrainingEducation($data, $educationTypeId);
            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('subject.lesson-title')->with("success", __("messages.record_successfully_created"));
        }

        return response()->json(["error" => __("error.generic_error")]);
    }

    /**
     * Get lesson titles
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function getLessonTitles(Request $request): Response|Application|ResponseFactory
    {
        $data = $request->all();
        $lessonTitles = null;

        if ($data["type"] == config('minesec.education_type.general')) {
            $request->validate(LessonTitleRules::getGeneralEducationLessonTitles());
            $lessonTitles = LessonTitle::getGeneralEducationSubjectLessonTitles($data["sub_system_id"], $data["education_type_id"], $data["class_id"], $data["subject_id"], $data["request_type"]);
        }

        if ($data["type"] == config('minesec.education_type.technical')) {
            $request->validate(LessonTitleRules::getTechnicalEducationLessonTitles());
            $lessonTitles = LessonTitle::getTechnicalEducationSubjectLessonTitles($data["sub_system_id"], $data["education_type_id"], $data["sub_education_type_id"], $data["specialty_id"], $data["specialty_class_id"], $data["subject_id"], $data["request_type"]);
        }

        if ($data["type"] == config('minesec.education_type.teacher_training')) {
            $request->validate(LessonTitleRules::getTeacherTrainingLessonTitle());
            $lessonTitles = LessonTitle::getTeacherTrainingSubjectLessonTitles($data["sub_system_id"], $data["education_type_id"], $data["level_id"], $data["level_class_id"], $data["subject_id"], $data["request_type"]);
        }

        if ($lessonTitles === null) {
            return api_response(150, 'No lessons found');
        }

        return api_response(100, 'ok', $lessonTitles);
    }

}
