<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Libraries\ConstantsLib;
use App\Models\Inspectorate;
use App\Models\Role;
use App\Models\User;
use App\RequestRules\Web\InspectorateRules;
use App\RequestRules\Web\UserRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class InspectorateController extends Controller
{

    /**
     * @var ConstantsLib
     */
    protected ConstantsLib $ConstantsLib;
    public function __construct(ConstantsLib $ConstantsLib)
    {
        $this->ConstantsLib = $ConstantsLib;
    }

    /**
     * Get inspectorate view
     *
     * @return Application|Factory|View
     */
    public function index(): Factory|View|Application
    {
        $inspectorates = Inspectorate::allInspectorates();

        return view("admin.inspectorate.index", compact("inspectorates"));
    }

    /**
     * Create inspectorate
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(InspectorateRules::create());

        $response = Inspectorate::createInspectorate($request->get("designation"));

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('inspectorate.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get edit view
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(InspectorateRules::edit($request->merge(["id" => $request->route("id")])));

        $inspectorates = Inspectorate::allInspectorates();
        $inspectorate = Inspectorate::findById($request->get("id"));

        return view("admin.inspectorate.edit", compact("inspectorates", "inspectorate"));
    }

    /**
     * Update inspectorate with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(InspectorateRules::update());
        $id = $request->{"id"};
        $designation = $request->{"designation"};

        $response = Inspectorate::updateInspectorate($id, $designation);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('inspectorate.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete inspectorate with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(InspectorateRules::delete());

        $response = Inspectorate::deleteInspectorate($request->get("id"));

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get personnel view
     *
     * @return Application|Factory|View
     */
    public function personnel(): View|Factory|Application
    {
        $personnels = User::getInspectoratePersonnel();
        $languages = $this->ConstantsLib->getLanguages();

        return view("Admin.inspectorate.personnel.list", compact("personnels", "languages"));
    }

    /**
     * Get create personnel view
     *
     * @param Request $request
     * @return View|Factory|Application
     */
    public function createPersonnel(Request $request): View|Factory|Application
    {
        $request->validate(InspectorateRules::edit($request->merge(["id" => $request->route("id")])));

        $genders = $this->ConstantsLib->getGender();
        $languages = $this->ConstantsLib->getLanguages();
        $inspectorate = Inspectorate::findById($request->get("id"));
        $inspectorateRoles = Role::getInspectorateRoles();
        $translatedRoles = $this->ConstantsLib->getRoles();

        return view("admin.inspectorate.personnel.index", compact("genders", "languages", "inspectorate", "inspectorateRoles", "translatedRoles"));
    }

    /**
     * Create inspectorate personnel
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function storePersonnel(Request $request): RedirectResponse
    {
        $request->validate(InspectorateRules::createPersonnel());

        $data = $request->all();

        $response = User::createInspectoratePersonnel($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('inspectorate.indexPersonnel')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Edit personnel
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function editPersonnel(Request $request): View|Factory|Application
    {
        $request->validate(UserRules::findUser($request->merge(["id" => $request->route("id")])));

        $personnel = User::findById($request->get("id"));
        $inspectorateRoles = Role::getInspectorateRoles();
        $genders = $this->ConstantsLib->getGender();
        $languages = $this->ConstantsLib->getLanguages();

        return view("admin.inspectorate.personnel.edit", compact("personnel", "inspectorateRoles", "genders", "languages"));
    }

    /**
     * Update personnel
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function updatePersonnel(Request $request): RedirectResponse
    {
        $request->validate(InspectorateRules::updatePersonnel());

        $data = $request->all();

        $response = User::updateInspectoratePersonnel($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('inspectorate.indexPersonnel')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete personnel
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deletePersonnel(Request $request): JsonResponse
    {
        $request->validate(UserRules::deleteUser());

        $response = User::deleteUser($request->get("id"));

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Block user
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function blockPersonnel(Request $request): JsonResponse
    {
        $request->validate(UserRules::blockUser());

        $response = User::blockUser($request->get("id"));

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.user_successfully_blocked")]);
    }

    public function unblockPersonnel(Request $request) {

    }

    public function subjects() {

        $inspectorates = Inspectorate::allInspectorates();

        return view("admin.inspectorate.subject.index", compact("inspectorates"));
    }
}
