<?php

namespace App\Services;

use App\Enums\PaymentGateWayEnum;
use App\Enums\TransactionDirectionEnum;
use App\Enums\TransactionStatusEnum;
use App\Enums\TransactionTypesNum;
use App\Enums\PasswordAndWalletPinChangeStatusEnum;
use App\Exceptions\InsufficientWalletBalanceException;
use App\Exceptions\InvalidPasswordException;
use App\Exceptions\WalletBlockedException;
use App\Exceptions\WalletCurrencyChangeException;
use App\Exceptions\WalletPinAlreadyUsedException;
use App\Exceptions\WalletRemainingPinAttemptsException;
use App\Models\Currency;
use App\Models\ExchangeRate;
use App\Models\ExternalAccount;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Wallet;
use App\Models\WalletPinChangeHistory;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

trait WalletService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      if (isset($data['wallet_number'])) {
         $data['wallet_number'] = mb_strtoupper($data['wallet_number']);
      }

      return Wallet::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      if (isset($data['wallet_number'])) {
         $data['wallet_number'] = mb_strtoupper($data['wallet_number']);
      }

      return tap($this)->update($data);
   }

   /**
    * CHeck wallet pin has already not been set
    *
    * @return bool
    */
   public function pinNotSet(): bool
   {
      return $this->{'wallet_pin'} == null;
   }

   /**
    * Decrements pin attempts left
    *
    * @return Builder|Model|int
    */
   public function decrementRemainingAttempts(): Model|Builder|int
   {
      $attemptsLeft = $this->{'pin_attempts_left'} - config("torryme.constants.default_increment_or_decrement");
      if ($attemptsLeft >= config("torryme.constants.default_zero_number")) {
         $this->updateService(['pin_attempts_left' => $attemptsLeft]);
      }

      return $attemptsLeft;
   }

   /**
    * Reset pin attempts left
    *
    * @return void
    */
   public function resetRemainingAttempts()
   {
      $this->updateService(['pin_attempts_left' => config("torryme.constants.max_pin_attempts")]);
   }

   /**
    * Block the user wallet
    */
   public function block()
   {
      return $this->updateService([
         'blocked_at' => now()
      ]);
   }

   /**
    * Check if is eligible to change wallet currency
    *
    * @return bool
    */
   public function canChangeCurrency(): bool
   {
      return $this->{"balance"} == config('torryme.constants.default_wallet_balance');
   }

   /**
    * Check balance is sufficient for the transaction
    *
    * @param float $amount
    * @throws InsufficientWalletBalanceException
    */
   public function inSufficientBalance(float $amount)
   {
      if ($amount > $this->{'balance'}) {
         throw  new InsufficientWalletBalanceException(__("errors.insufficient_wallet_balance"));
      }
   }

   /**
    * Check if source and destination currencies are the same
    *
    * @param int $sourceCurrency
    * @param int $destinationCurrency
    * @return bool
    */
   public function isSameCurrency(int $sourceCurrency, int $destinationCurrency): bool
   {
      return $sourceCurrency === $destinationCurrency;
   }

   /**
    * Get wallet balance
    *
    * @return mixed
    */
   public function getWalletBalance(): mixed
   {
      return $this->{"balance"};
   }

   /**
    * Debit wallet
    *
    * @param float $debitAmount
    * @return mixed
    */
   public function debitWallet(float $debitAmount): mixed
   {
      $amountAfterDebit = $this->{"balance"} - $debitAmount;
      $this->updateService(['balance' => $amountAfterDebit]);

      return $this->{"balance"};
   }

   /**
    * Credit wallet
    *
    * @param float $creditAmount
    * @return mixed
    */
   public function creditWallet(float $creditAmount): mixed
   {
      $amountAfterCredit = $this->{"balance"} + $creditAmount;
      $this->updateService(['balance' => $amountAfterCredit]);

      return $this->{"balance"};
   }

   /**
    * Decrement wallet pin attempts
    *
    * @param string $walletPin
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    */
   public function verifyWalletPin(string $walletPin)
   {
      if (!$this->pinNotSet()) {
         if (!$this->checkWalletPin($walletPin)) {
            if ($this->{"pin_attempts_left"} == config("torryme.constants.default_zero_number")) {
               $this->block();
               throw  new WalletBlockedException(message: __('errors.wallet_blocked'));
            } else {
               $attemptsLeft = $this->decrementRemainingAttempts();
            }
            throw new  WalletRemainingPinAttemptsException(
               message: trans_choice('errors.wallet_pin_incorrect', $attemptsLeft, ['value' => $attemptsLeft])
            );
         } else {
            $this->resetRemainingAttempts();
         }
      } else {
         throw new WalletRemainingPinAttemptsException(__("errors.wallet_pin_not_yet_set"));
      }
   }

   /**
    * Create user wallet
    *
    * @param User $user
    * @param Currency|null $currency
    * @return Builder|Model
    */
   public static function createUserWallet(User $user, Currency $currency = null): Model|Builder
   {
      $defaultCurrency = Currency::findByCode(config('torryme.constants.default_currency_code'));

      return Wallet::store([
         'balance' => config('torryme.constants.default_wallet_balance'),
         'wallet_pin' => null,
         'pin_attempts_left' => config('torryme.constants.max_pin_attempts'),
         'wallet_number' => generate_wallet_number('' . $user->{'id'}),
         'user_id' => $user->{'id'},
         'currency_id' => $currency?->{'id'} ?? $defaultCurrency->{'id'},
      ]);
   }

   /**
    * Find by wallet number
    *
    * @param string $walletNumber
    * @param $authUserWalletNumber
    * @return Model|Builder|null
    */
   public static function findByWalletNumber(string $walletNumber, $authUserWalletNumber): Model|Builder|null
   {
      try {
         $result = null;

         $walletSearch = Wallet::query()->where("wallet_number", $walletNumber)
            ->whereNot(function (Builder $builder) use ($authUserWalletNumber) {
               $builder->where('wallet_number', $authUserWalletNumber);
            })->first();

         $result = $walletSearch;
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::findByWalletNumber');
      }
      return $result;

   }

   /**
    * Find by id
    *
    * @param int $walletId
    * @return Builder|Builder[]|Collection|Model|null
    */
   public static function findById(int $walletId): Model|Collection|Builder|array|null
   {
      return Wallet::query()->find($walletId);
   }

   /**
    * Set wallet pin
    *
    * @param string $walletPin
    * @return null
    */
   public function setPin(string $walletPin)
   {
      DB::beginTransaction();
      $wallet = $this;
      $user = $wallet->{"user"};

      try {
         /** @var Wallet $walletUpdate */
         $walletUpdate = $wallet->updateService(["wallet_pin" => Hash::make($walletPin)]);

         // Set wallet pin change history
         WalletPinChangeHistory::store([
            "old_wallet_pin" => Hash::make($walletPin),
            "new_wallet_pin" => Hash::make($walletPin),
            "status" => PasswordAndWalletPinChangeStatusEnum::active->value,
            "wallet_id" => $wallet->{"id"},
            "user_id" => $user->{"id"},
         ]);

         DB::commit();
         $result = $walletUpdate->refresh()->unsetRelation('user');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::setPin');
         DB::rollBack();
         $result = null;
      }

      return $result;
   }

   /**
    * Update wallet pin
    *
    * @param array $data
    * @return Wallet|null
    * @throws WalletBlockedException
    * @throws WalletPinAlreadyUsedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public function updatePin(array $data): ?Wallet
   {
      $wallet = $this;
      $user = $wallet->{"user"};

      //Check current pin match
      $wallet->verifyWalletPin($data["old_wallet_pin"]);

      //Get wallet change histories
      $walletChangeHistories = $wallet->{"walletPinChangeHistory"};

      foreach ($walletChangeHistories as $walletChangeHistory) {
         if ($walletChangeHistory->checkIfWalletPinAlreadyUsed($data["wallet_pin"])) {
            throw new WalletPinAlreadyUsedException(__("errors.wallet_pin_already_used"));
         }
      }

      DB::beginTransaction();
      try {
         //Update wallet pin
         $wallet->updateService(["wallet_pin" => Hash::make($data["wallet_pin"])]);

         //Get last wallet pin change history
         $lastWalletPinChaneHistory = WalletPinChangeHistory::getLastChangeHistory($wallet->{"id"});

         //Change status of last history to inactive
         $lastWalletPinChaneHistory->updateService(["status" => PasswordAndWalletPinChangeStatusEnum::inactive->value]);

         //Create new wallet pin change history
         WalletPinChangeHistory::store([
            "old_wallet_pin" => Hash::make($data["old_wallet_pin"]),
            "new_wallet_pin" => Hash::make($data["wallet_pin"]),
            "status" => PasswordAndWalletPinChangeStatusEnum::active->value,
            "wallet_id" => $wallet->{"id"},
            "user_id" => $user->{"id"},
         ]);

         DB::commit();
         $result = $wallet->refresh()->unsetRelation('user');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::updatePin');
         DB::rollBack();
         throw  $exception;
      }

      return $result;
   }

   /**
    * Transfer funds to internal account
    *
    * @param array $data
    * @return Wallet|string|null
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    */
   public function transferToAccount(array $data): Wallet|string|null
   {
      $sourceWallet = $this;

      //Check current pin match
      $sourceWallet->verifyWalletPin($data["wallet_pin"]);

      DB::beginTransaction();
      try {
         //Check for insufficient balance
         $sourceWallet->inSufficientBalance($data["source_amount"]);

         // Get recipient user
         $recipient = User::findById($data["recipient_id"]);

         //Get recipient wallet
         /** @var Wallet $destinationWallet */
         $destinationWallet = $recipient->{"wallet"};
         $sourceAmount = $data["source_amount"];

         //Get source wallet balance before debit
         $sourceOldBalance = $sourceWallet->getWalletBalance();

         //Debit source wallet and return balance after debit
         $sourceNewBalance = $sourceWallet->debitWallet($sourceAmount);

         $transactionType = TransactionTypesNum::transfer->value;
         $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

         //Check if source and destination currencies are the same
         if ($sourceWallet->isSameCurrency($sourceWallet->{"currency_id"}, $destinationWallet->{"currency_id"})) {

            //Get destination wallet old balance
            $destinationOldBalance = $destinationWallet->getWalletBalance();

            //Credit destination wallet and get new wallet balance after credit
            $destinationNewBalance = $destinationWallet->creditWallet($sourceAmount);

            //Store transaction details
            Transaction::store([
               "type" => $transactionType,
               "status" => TransactionStatusEnum::completed->value,
               "source_amount" => $sourceAmount,
               "source_old_balance" => $sourceOldBalance,
               "source_new_balance" => $sourceNewBalance,
               "destination_amount" => $sourceAmount,
               "destination_old_balance" => $destinationOldBalance,
               "destination_new_balance" => $destinationNewBalance,
               "exchange_rate" => config("torryme.constants.same_currency_exchange_rate"),
               "transaction_reference" => transaction_reference($transactionTypeInitial),
               "payment_gateway" => PaymentGateWayEnum::internal->value,
               "source_wallet_id" => $sourceWallet->{"id"},
               "source_currency_id" => $sourceWallet->{"currency_id"},
               "destination_wallet_id" => $destinationWallet->{"id"},
               "destination_currency_id" => $destinationWallet->{"currency_id"},
            ]);

         } else {
            $exchangeRate = ExchangeRate::getExchangeRate($sourceWallet->{"currency_id"}, $destinationWallet->{"currency_id"});
            $destinationAmount = $sourceAmount * $exchangeRate->{"exchange_rate"};

            //Get destination wallet balance before credit
            $destinationOldBalance = $destinationWallet->getWalletBalance();

            //Credit destination wallet and get new wallet balance after credit
            $destinationNewBalance = $destinationWallet->creditWallet($destinationAmount);

            //Store transaction details
            Transaction::store([
               "type" => $transactionType,
               "status" => TransactionStatusEnum::completed->value,
               "source_amount" => $sourceAmount,
               "source_old_balance" => $sourceOldBalance,
               "source_new_balance" => $sourceNewBalance,
               "destination_amount" => $destinationAmount,
               "destination_old_balance" => $destinationOldBalance,
               "destination_new_balance" => $destinationNewBalance,
               "exchange_rate" => $exchangeRate->{"exchange_rate"},
               "transaction_reference" => transaction_reference($transactionTypeInitial),
               "payment_gateway" => PaymentGateWayEnum::internal->value,
               "source_wallet_id" => $sourceWallet->{"id"},
               "source_currency_id" => $sourceWallet->{"currency_id"},
               "destination_wallet_id" => $destinationWallet->{"id"},
               "destination_currency_id" => $destinationWallet->{"currency_id"},
            ]);
         }

         DB::commit();
         $result = $this->refresh()->load('currency');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::transferToAccount');
         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Deposit to account
    *
    * @param array $data
    * @return Wallet|null
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    * @throws Exception
    */
   public function accountDeposit(array $data): ?Wallet
   {
      $wallet = $this;

      //Check current pin match
      $wallet->verifyWalletPin($data["wallet_pin"]);

      DB::beginTransaction();
      try {
         //Get external account by id
         $externalAccount = ExternalAccount::findById($data["external_account_id"]);

         //Get wallet balance before credit
         $oldDestinationBalance = $wallet->getWalletBalance();

         //Credit wallet and return new balance after credit
         $newDestinationBalance = $wallet->creditWallet($data["deposit_amount"]);

         $transactionType = TransactionTypesNum::deposit->value;
         $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

         //Create transaction history
         Transaction::store([
            "type" => $transactionType,
            "status" => TransactionStatusEnum::completed->value,
            "source_amount" => $data["deposit_amount"],
            "source_old_balance" => null,
            "source_new_balance" => null,
            "destination_amount" => $data["deposit_amount"],
            "destination_old_balance" => $oldDestinationBalance,
            "destination_new_balance" => $newDestinationBalance,
            "exchange_rate" => config("torryme.constants.same_currency_exchange_rate"),
            "transaction_reference" => transaction_reference($transactionTypeInitial),
            "payment_gateway" => $externalAccount->{"payment_gateway"},
            "source_wallet_id" => null,
            "source_currency_id" => null,
            "destination_wallet_id" => $wallet->{"id"},
            "destination_currency_id" => $wallet->{"currency_id"},
            "external_account_source_id" => $externalAccount->{"id"},
            "external_account_destination_id" => null,
         ]);

         // TODO platform wallet withdrawal

         DB::commit();
         $result = $wallet->refresh()->load('currency');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::accountDeposit');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Withdrawal from account
    *
    * @param array $data
    * @return Wallet|null
    * @throws InsufficientWalletBalanceException
    * @throws WalletBlockedException
    * @throws WalletRemainingPinAttemptsException
    */
   public function accountWithdrawal(array $data): ?Wallet
   {
      $wallet = $this;

      //Check current pin match
      $wallet->verifyWalletPin($data["wallet_pin"]);

      DB::beginTransaction();
      try {
         //Check for sufficient balance
         $wallet->inSufficientBalance($data["withdrawal_amount"]);

         //Get external account by id
         $externalAccount = ExternalAccount::findById($data["external_account_id"]);

         //Get wallet balance before debit
         $oldSourceBalance = $wallet->getWalletBalance();

         //Debit wallet and return new balance after debit
         $newSourceBalance = $wallet->debitWallet($data["withdrawal_amount"]);

         $transactionType = TransactionTypesNum::withdrawal->value;
         $transactionTypeInitial = strtoupper(substr($transactionType, 0, 1));

         //Create transaction history
         Transaction::store([
            "type" => $transactionType,
            "status" => TransactionStatusEnum::completed->value,
            "source_amount" => $data["withdrawal_amount"],
            "source_old_balance" => $oldSourceBalance,
            "source_new_balance" => $newSourceBalance,
            "destination_amount" => $data["withdrawal_amount"],
            "destination_old_balance" => null,
            "destination_new_balance" => null,
            "exchange_rate" => config("torryme.constants.same_currency_exchange_rate"),
            "transaction_reference" => transaction_reference($transactionTypeInitial),
            "payment_gateway" => $externalAccount->{"payment_gateway"},
            "source_wallet_id" => $wallet->{"id"},
            "source_currency_id" => $wallet->{"currency_id"},
            "destination_wallet_id" => null,
            "destination_currency_id" => null,
            "external_account_source_id" => null,
            "external_account_destination_id" => $externalAccount->{"id"},
         ]);

         // TODO platform wallet withdrawal

         DB::commit();
         $result = $wallet->refresh()->load('currency');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::accountWithdrawal');
         DB::rollBack();

         throw $exception;
      }

      return $result;
   }

   /**
    * Get users transactions
    *
    * @param string|null $type
    * @param int|null $pageNumber
    * @return array|null
    */
   public function transactions(string $type = null, int $pageNumber = null): ?array
   {
      try {
         $wallet = $this;
         $usersTransactions = Transaction::allUserTransactions($wallet, $type, $pageNumber);
         $totalPages = $usersTransactions->total() == config("torryme.constants.total_pages_for_no_transactions") ? config("torryme.constants.total_pages_for_no_transactions") : $usersTransactions->lastPage();

         $arrayResults = array();
         foreach ($usersTransactions as $usersTransaction) {
            $reference = null;
            $action = null;
            $amount = null;
            $currency = null;
            $status = null;
            $direction = null;

            $externalAccountSourceId = $usersTransaction->{"external_account_source_id"};
            $destinationWalletId = $usersTransaction->{"destination_wallet_id"};

            $externalAccountDestinationId = $usersTransaction->{"external_account_destination_id"};
            $sourceWalletId = $usersTransaction->{"source_wallet_id"};

            //Deposit transactions
            if ($externalAccountSourceId != null && $destinationWalletId == $wallet->{"id"}) {
               $reference = match ($usersTransaction->{"payment_gateway"}) {
                  config("torryme.payment_gateways.momo"), config("torryme.payment_gateways.orange_money"), config("torryme.payment_gateways.eu_money") => $usersTransaction->{"externalSourceAccount"}->{"telephone"},
                  config("torryme.payment_gateways.master_card"), config("torryme.payment_gateways.visa") => self::maskCardNumber($usersTransaction->{"externalSourceAccount"}->{"card_number"}),
                  config("torryme.payment_gateways.paypal") => $usersTransaction->{"externalSourceAccount"}->{"email"},
                  default => PaymentGateWayEnum::internal->value,
               };

               $action = __("messages.transaction_actions.deposit_through") . " " . $usersTransaction->{"payment_gateway"};
               $amount = $usersTransaction->{"source_amount"};
               $currency = $usersTransaction->{"destinationCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::in->value;
            }

            //Withdrawal transactions
            if ($externalAccountDestinationId != null && $sourceWalletId == $wallet->{"id"}) {
               $reference = match ($usersTransaction->{"payment_gateway"}) {
                  config("torryme.payment_gateways.momo"), config("torryme.payment_gateways.orange_money"), config("torryme.payment_gateways.eu_money") => $usersTransaction->{"externalDepositAccount"}->{"telephone"},
                  config("torryme.payment_gateways.master_card"), config("torryme.payment_gateways.visa") => self::maskCardNumber($usersTransaction->{"externalDepositAccount"}->{"card_number"}),
                  config("torryme.payment_gateways.paypal") => $usersTransaction->{"externalDepositAccount"}->{"email"},
                  default => PaymentGateWayEnum::internal->value,
               };

               $action = __("messages.transaction_actions.withdrawal_through") . " " . $usersTransaction->{"payment_gateway"};
               $amount = $usersTransaction->{"source_amount"};
               $currency = $usersTransaction->{"sourceCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::out->value;
            }

            //Send Internal Transfer
            if ($sourceWalletId == $wallet->{"id"} && $usersTransaction->{"type"} == TransactionTypesNum::transfer->value) {
               $userDetails = $usersTransaction->{"destinationWallet"}->{"user"}->{"userDetail"};
               $reference = $userDetails->{"first_name"} . " " . $userDetails->{"last_name"};
               $action = __("messages.transaction_actions.transferred_to") . " " . "@" . $usersTransaction->{"destinationWallet"}->{"user"}->{"user_name"};
               $amount = $usersTransaction->{"source_amount"};
               $currency = $usersTransaction->{"sourceCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::out->value;
            }

            //Receive Internal Transfer
            if ($destinationWalletId == $wallet->{"id"} && $usersTransaction->{"type"} == TransactionTypesNum::transfer->value) {
               $userDetails = $usersTransaction->{"sourceWallet"}->{"user"}->{"userDetail"};
               $reference = $userDetails->{"first_name"} . " " . $userDetails->{"last_name"};
               $action = __("messages.transaction_actions.received_from") . " " . "@" . $usersTransaction->{"sourceWallet"}->{"user"}->{"user_name"};
               $amount = $usersTransaction->{"destination_amount"};
               $currency = $usersTransaction->{"destinationCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::in->value;
            }

            //Client payment
            if ($sourceWalletId == $wallet->{"id"} && $usersTransaction->{"type"} == TransactionTypesNum::payment->value) {
               $userDetails = $usersTransaction->{"destinationWallet"}->{"user"}->{"business"};
               $reference = $userDetails->{"designation"};
               $action = __("messages.transaction_actions.payment_to") . " " . "@" . $usersTransaction->{"destinationWallet"}->{"user"}->{"user_name"};
               $amount = $usersTransaction->{"source_amount"};
               $currency = $usersTransaction->{"sourceCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::out->value;
            }

            //Client payment refund
            if ($sourceWalletId == $wallet->{"id"} && $usersTransaction->{"type"} == TransactionTypesNum::refund->value) {
               $userDetails = $usersTransaction->{"sourceWallet"}->{"user"}->{"userDetail"};
               $reference = $userDetails->{"first_name"} . " " . $userDetails->{"last_name"};
               $action = __("messages.transaction_actions.refund_to") . " " . "@" . $usersTransaction->{"sourceWallet"}->{"user"}->{"user_name"};
               $amount = $usersTransaction->{"source_amount"};
               $currency = $usersTransaction->{"sourceCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::in->value;
            }

            //Business payment
            if ($destinationWalletId == $wallet->{"id"} && $usersTransaction->{"type"} == TransactionTypesNum::payment->value) {
               $userDetails = $usersTransaction->{"sourceWallet"}->{"user"}->{"userDetail"};
               $reference = $userDetails->{"first_name"} . " " . $userDetails->{"last_name"};
               $action = __("messages.transaction_actions.payment_from") . " " . "@" . $usersTransaction->{"sourceWallet"}->{"user"}->{"user_name"};
               $amount = $usersTransaction->{"destination_amount"};
               $currency = $usersTransaction->{"destinationCurrency"}->{"code"};
               $status = $usersTransaction->{"status"};
               $direction = TransactionDirectionEnum::in->value;
            }

            $arrayResults[] = array(
               "id" => $usersTransaction->{"id"},
               "action" => $action,
               'amount' => $amount,
               'currency' => $currency,
               'reference' => $reference,
               "status" => $status,
               "time" => $usersTransaction->{"created_at"},
               "direction" => $direction,
            );
         }

         $finalData = array(
            "pages" => $totalPages,
            "transactions" => $arrayResults,
         );

         $result = $finalData;
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletService::transactions');
         $result = null;
      }

      return $result;
   }

   /**
    * Mask credit card number
    *
    * @param $number
    * @param string $maskingCharacter
    * @return string
    */
   public static function maskCardNumber($number, string $maskingCharacter = '.'): string
   {
      return substr($number, 0, 4) . str_repeat($maskingCharacter, strlen($number) - 8) . substr($number, -4);
   }
}
