<?php

namespace App\Services;

use App\Enums\AccountPrivacyEnum;
use App\Enums\DocumentTypeEnum;
use App\Enums\EmailAndTelephoneTypeEnum;
use App\Enums\GenericStatusEnum;
use App\Enums\PasswordAndWalletPinChangeStatusEnum;
use App\Enums\UserTypeEnum;
use App\Exceptions\EmailAlreadyExistException;
use App\Exceptions\InvalidPasswordException;
use App\Exceptions\PasswordAlreadyUsedException;
use App\Exceptions\PasswordConfirmationException;
use App\Exceptions\TelephoneAlreadyExistException;
use App\Exceptions\TelephoneNotVerifiedException;
use App\Exceptions\UploadFileException;
use App\Exceptions\UserNameAlreadyExistException;
use App\Exceptions\WalletCurrencyChangeException;
use App\Http\Controllers\Api\Auth\AuthenticateController;
use App\Models\AccountNotificationSetting;
use App\Models\AccountPrivacySetting;
use App\Models\Business;
use App\Models\BusinessSubCategory;
use App\Models\Country;
use App\Models\Currency;
use App\Models\DeletedAccount;
use App\Models\Follower;
use App\Models\Order;
use App\Models\OtpVerification;
use App\Models\PasswordChangeHistory;
use App\Models\Post;
use App\Models\RealtimeMessage;
use App\Models\Save;
use App\Models\Share;
use App\Models\User;
use App\Models\UserDetail;
use App\Models\UserDevice;
use App\Models\UserEmail;
use App\Models\UserTelephone;
use App\Models\UserVerification;
use App\Models\Wallet;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

trait UserService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return User::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    *
    * @return bool
    */
   public function noRequestSubmitted(): bool
   {
      return $this->{"verification_requested_at"} == null && $this->{"verified_at"} == null && $this->{"declined_at"} == null;
   }

   /**
    * Find by id
    *
    * @param int $id
    * @return Builder|Builder[]|Collection|Model|null
    */
   public static function findById(int $id): Model|Collection|Builder|array|null
   {
      return User::query()->find($id);
   }

   /**
    * Check user password
    *
    * @throws InvalidPasswordException
    */
   public function checkUserPassword(string $password)
   {
      if (!$this->verifyPassword($password)) {
         throw new InvalidPasswordException(__("errors.invalid_password"));
      }
   }

   /**
    * Get user device
    *
    * @return mixed
    */
   public function lastUserDevice(): mixed
   {
      $user = $this;

      return UserDevice::lastUserDevice($user);
   }

   /**
    * Find user by email, telephone or username
    *
    * @param string $username
    * @return Builder|Model|null
    */
   public static function oneStepAuthentication(string $username): Model|Builder|null
   {
      return
         User::query()
            ->where('telephone', $username)
            ->first();
   }

   /**
    * Check if username is already in use
    *
    * @param string $username
    * @return bool
    */
   public static function userNameExist(string $username): bool
   {
      return User::query()->where('user_name', $username)->exists();
   }

   /**
    * Check if telephone number is already in use
    *
    * @param string $telephone
    * @return bool
    */
   public static function telephoneNumberExist(string $telephone): bool
   {
      return User::query()->where('telephone', $telephone)->exists();
   }

   /**
    * Find by telephone
    *
    * @param string $telephone
    * @return Builder|Model|null
    */
   public static function findByPhone(string $telephone): Model|Builder|null
   {
      return User::query()->where('telephone', $telephone)->first();
   }

   /**
    * Check if email is already in use
    *
    * @param string $email
    * @return bool
    */
   public static function emailAddressExist(string $email): bool
   {
      return User::query()->where('email', $email)->exists();
   }

   /**
    * Create new user
    *
    * @param array $data
    * @param array $files
    * @return Model|Builder|null
    * @throws TelephoneNotVerifiedException
    * @throws UploadFileException
    */
   public static function registerUser(array $data, array $files = []): Model|Builder|null
   {
      $result = null;
      DB::beginTransaction();

      try {
         // Checking otp verification
         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone($data['telephone'], null);

         if ($otpEntry === null || !$otpEntry->isValidated()) {
            if (isset($data['telephone'])) {
               throw new TelephoneNotVerifiedException(__('errors.telephone_not_verified'));
            }
         }

         // User details ...
         $userDetailData = Arr::only($data, UserDetail::creationAttributes());
         $userDetail = UserDetail::store($userDetailData);

         $userData = Arr::only($data, User::creationAttributes());
         $userData['user_detail_id'] = $userDetail->{'id'};
         $userData['user_type'] = UserTypeEnum::personal->value;
         $userData['password'] = Hash::make($data['password']);
         $userData['language_code'] = mb_strtolower(
            request()->{'lang'} ?? config('torryme.locales_keys.en')
         );

         // Try to get country
         if (isset($userData['telephone']) && filled($userData['telephone'])) {
            $country = Country::handleAllCountries($userData['telephone']);
            $userData['country_id'] = $country?->{'id'};
         } else if (filled($userData['country_code'])) {
            $country = Country::createOrUpdate($userData['country_code']);
            if ($country !== null) {
               $userData['country_id'] = $country->{'id'};
            }
         }

         // Store user ...
         /** @var User $user */
         $user = User::store($userData);

         // Profile picture ...
         if (isset($files['profile_picture'])) {
            $docBasePath = sprintf(config('torryme.paths.docs'), User::$prefixDir . $user->{'id'});
            $path = upload_image(
               DocumentTypeEnum::avatar->value,
               $docBasePath,
               $files['profile_picture']
            );

            if (!filled($path)) {
               throw new UploadFileException(__('errors.unable_to_upload_avatar'));
            }

            $user->updateService(['profile_picture' => $path]);
         }

         // Update opt verification
         $otpEntry->updateService([
            'user_id' => $user->{'id'}
         ]);

         // Device ...
         UserDevice::store(array_merge(
            Arr::only($data, UserDevice::creationAttributes()),
            ['user_id' => $user->{'id'}]
         ));

         // Create Primary telephone
         if (isset($data['telephone'])) {
            UserTelephone::store([
               "telephone" => $data['telephone'],
               "type" => EmailAndTelephoneTypeEnum::primary->value,
               "user_id" => $user->{'id'},
               "verified_at" => now()
            ]);
         }

         // Create password change history
         PasswordChangeHistory::store([
            "old_password" => $userData['password'],
            "new_password" => $userData['password'],
            "status" => PasswordAndWalletPinChangeStatusEnum::active->value,
            "user_id" => $user->{'id'},
         ]);

         // Create account privacy setting
         AccountPrivacySetting::store([
            "enable_privacy" => GenericStatusEnum::enable->value,
            "enable_2factor_auth" => GenericStatusEnum::disable->value,
            "who_can_comment" => AccountPrivacyEnum::everyOne->value,
            "who_can_mention" => AccountPrivacyEnum::everyOne->value,
            "who_can_see_follower_list" => AccountPrivacyEnum::everyOne->value,
            "who_can_invite_you_by_direct_message" => AccountPrivacyEnum::everyOne->value,
            "who_can_invite_you_into_community" => AccountPrivacyEnum::everyOne->value,
            "who_can_download_post" => AccountPrivacyEnum::everyOne->value,
            "user_id" => $user->{'id'},
         ]);

         // Create account notification settings
         AccountNotificationSetting::store([
            "enable_push_notification" => GenericStatusEnum::enable->value,
            "new_orders" => GenericStatusEnum::enable->value,
            "likes" => GenericStatusEnum::enable->value,
            "comments" => GenericStatusEnum::enable->value,
            "shares" => GenericStatusEnum::enable->value,
            "reviews" => GenericStatusEnum::enable->value,
            "direct_messages" => GenericStatusEnum::enable->value,
            "post_from_subscribed_accounts" => GenericStatusEnum::enable->value,
            "live_stream_from_subscribed_accounts" => GenericStatusEnum::enable->value,
            "disputes" => GenericStatusEnum::enable->value,
            "deposits" => GenericStatusEnum::enable->value,
            "withdrawals" => GenericStatusEnum::enable->value,
            "transfers" => GenericStatusEnum::enable->value,
            "new_subscribers" => GenericStatusEnum::enable->value,
            "post_mentions" => GenericStatusEnum::enable->value,
            "user_id" => $user->{'id'},
         ]);

         // Create wallet ...
         Wallet::createUserWallet($user);

         $result = $user->refresh();
         DB::commit();
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::registerUser');

         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Edit profile
    *
    * @throws UploadFileException|UserNameAlreadyExistException
    */
   public function editProfile(array $data): User
   {
      $user = $this;
      /** @var Business $business */
      $business = $user->{"business"};

      DB::beginTransaction();
      try {
         if (isset($data["first_name"])) {
            $user->{"userDetail"}->updateService([
               "first_name" => $data["first_name"],
            ]);
         }

         if (isset($data["last_name"])) {
            $user->{"userDetail"}->updateService([
               "last_name" => $data["last_name"],
            ]);
         }

         if (isset($data["date_of_birth"])) {
            $user->{"userDetail"}->updateService([
               "date_of_birth" => $data["date_of_birth"],
            ]);
         }

         if (isset($data['avatar'])) {
            if ($data['avatar'] == null) {
               $user->updateService(['profile_picture' => null]);
            } else {
               $docBasePath = sprintf(config('torryme.paths.docs'), User::$prefixDir . $user->{'id'});
               $path = upload_image(
                  DocumentTypeEnum::avatar->value,
                  $docBasePath,
                  $data['avatar']
               );

               if (!filled($path)) {
                  throw new UploadFileException(__('errors.unable_to_upload_avatar'));
               }
               //TODO delete old profile photo after uploading new one
               $user->updateService(['profile_picture' => $path]);
            }
         }

         if ($business != null) {
            if (isset($data["business_designation"])) {
               $business->updateService([
                  "designation" => $data["business_designation"],
               ]);
            }

            if (isset($data["description "])) {
               $business->updateService([
                  "description" => $data["description "],
               ]);
            }

         } else {
            if (isset($data["bio"])) {
               $user->updateService([
                  "bio" => $data["bio"],
               ]);
            }
         }

         if (isset($data["user_name"])) {
            //Check username does not already exist and if it does, it belongs to the Auth User
            User::checkUserUserNameUniqueness($data["user_name"]);
            $user->updateService([
               "user_name" => $data["user_name"],
            ]);
         }

         DB::commit();
         $result = AuthenticateController::responseWithUser($user->refresh());
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::editProfile');

         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Create user interests
    *
    * @param array $data
    * @return User|null
    */
   public static function createUserInterest(array $data): ?User
   {
      /** @var User $user */
      $user = auth()->user();

      try {
         $interestExist = BusinessSubCategory::checkInterestExist($data);
         $user->userInterest()->attach($interestExist);

         $result = $user->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::createUserInterest');
         $result = null;
      }

      return $result;
   }

   /**
    * Configure account
    *
    * @param array $data
    * @param Wallet $wallet
    * @param User $user
    * @return User|null
    * @throws WalletCurrencyChangeException
    */
   public static function configureAccount(array $data, Wallet $wallet, User $user): ?User
   {
      DB::beginTransaction();

      try {
         // Check if language code is present
         if (isset($data["language_code"])) {
            // Update language code
            $user->updateService(["language_code" => $data["language_code"]]);
         }

         // Check if currency code is present
         if (isset($data['currency_code'])) {

            //Check user can change currency
            if (!$wallet->canChangeCurrency()) {
               throw new WalletCurrencyChangeException(__("errors.invalid_wallet_currency_change"));
            }

            // Get currency by code
            $currency = Currency::findByCode($data['currency_code']);

            // Update wallet currency
            $wallet->updateService(["currency_id" => $currency->{"id"}]);
         }

         //CHeck if country code is present
         if (isset($data['country_code'])) {
            // Get country by code
            $country = Country::findByCountryCode($data['country_code']);

            // Update language code
            $user->updateService(["country_id" => $country->{"id"}]);
         }

         DB::commit();

         $result = AuthenticateController::responseWithUser($user);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::configureAccount');
         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * CHeck user verification
    *
    * @return Builder|Model|null
    */
   public function userVerificationStatus(): Model|Builder|null
   {
      return UserVerification::checkStatus($this);
   }

   /**
    * Compute user visitor profile
    *
    * @return array
    */
   public function computeProfileForVisitor(): array
   {
      /** @var User $authUser */
      $authUser = auth()->user();
      $user = $this;

      $isFollower = $authUser?->isFollower($user->{'id'});
      $canSeeFollowerList = $authUser->whoCanSeeFollowerList($user);

      return array(
         "user" => $user->{"business"} == null ? $user->load("userDetail") : $user->load(['userDetail', 'business.businessCategory']),
         "statistics" => $this->computeProfileStatistics(),
         "posts" => Post::userMostRecentPost($user),
         "shares" => Share::userSharedPosts($user),
         "saves" => Save::userSavedPosts($user),

         "is_follower" => $isFollower ? config("torryme.constants.default_one_number") : config("torryme.constants.default_zero_number"),
         "can_see_follower_list" => $canSeeFollowerList ? config("torryme.constants.default_one_number") : config("torryme.constants.default_zero_number"),
      );
   }

   /**
    * Compute user profile statistics
    *
    * @return array
    */
   public function computeProfileStatistics(): array
   {
      return array(
         "saves" => (int)Post::getTotalUserPostSaves($this->{"id"}),
         "shares" => (int)Post::getTotalUserPostShares($this->{"id"}),
         "followers" => Follower::getNumberOfFollowers($this->{"id"}),
         "following" => Follower::getNumberOfFollowings($this->{"id"}),
         "likes" => (int)Post::getTotalUserPostLikes($this->{"id"}),
         "reviews" => $this->{'business'} !== null ? $this->{'business'}->totalReviews() : 0,
      );
   }

   /**
    * Check user username uniqueness
    *
    * @param string $userName
    * @return bool|null
    * @throws UserNameAlreadyExistException
    */
   public static function checkUserUserNameUniqueness(string $userName): ?bool
   {
      /** @var User $user */
      $user = auth()->user();

      $userNameExist = self::userNameExist($userName);
      if ($userNameExist) {
         $isTheSame = strcmp($user->{"user_name"}, $userName);
         if ($isTheSame == 0) {
            return true;
         } else {
            throw new UserNameAlreadyExistException(__("errors.user_name_already_taken"));
         }
      }

      return false;
   }

   /**
    * Check telephone number uniqueness
    *
    * @param string $telephone
    * @return bool
    * @throws TelephoneAlreadyExistException
    */
   public static function checkPhoneNumberUniqueness(string $telephone): bool
   {
      /** @var User $user */
      $user = auth()->user();

      $telephoneExist = self::telephoneNumberExist($telephone);
      if ($telephoneExist) {
         $telephoneBelongsToUser = strcmp($user->{"telephone"}, $telephone);
         if ($telephoneBelongsToUser == 0) {
            return true;
         } else {
            throw new TelephoneAlreadyExistException(__("errors.phone_number_already_taken"));
         }
      }

      return false;
   }

   /**
    * Check email address uniqueness
    *
    * @param string $email
    * @return bool
    * @throws EmailAlreadyExistException
    */
   public static function checkEmailAddressUniqueness(string $email): bool
   {
      /** @var User $user */
      $user = auth()->user();

      $emailExist = self::emailAddressExist($email);
      if ($emailExist) {
         $emailBelongsToUser = strcmp($user->{"email"}, $email);
         if ($emailBelongsToUser == 0) {
            return true;
         } else {
            throw new EmailAlreadyExistException(__("errors.email_address_already_taken"));
         }
      }

      return false;
   }

   /**
    * Check telephone uniqueness
    *
    * @param string $telephone
    * @return false
    * @throws TelephoneAlreadyExistException
    */
   public static function checkPhoneNumberIsUniqueAndNotUsedByCurrentUser(string $telephone): bool
   {
      /** @var User $user */
      $user = auth()->user();

      $telephoneExist = UserTelephone::checkNumberExist($telephone);
      if ($telephoneExist) {
         $userTelephones = $user->{"userTelephones"};
         if (sizeof($userTelephones) !== 0) {
            foreach ($userTelephones as $userTelephone) {
               if ($telephone === $userTelephone->{"telephone"}) {
                  throw new TelephoneAlreadyExistException(__("errors.number_already_in_use_by_you"));
               }
            }
         }

         throw new TelephoneAlreadyExistException(__("errors.phone_number_already_taken"));
      }

      return false;
   }

   /**
    * Check email uniqueness
    *
    * @param string $email
    * @return false
    * @throws EmailAlreadyExistException
    */
   public static function checkEmailIsUniqueAndNotUsedByCurrentUser(string $email): bool
   {
      /** @var User $user */
      $user = auth()->user();

      $emailExist = UserEmail::checkEmailExist($email);
      if ($emailExist) {
         $userEmails = $user->{"userEmails"};
         if (sizeof($userEmails) != 0) {
            foreach ($userEmails as $userEmail) {
               if ($email == $userEmail->{"email"}) {
                  throw new EmailAlreadyExistException(__("errors.email_already_in_use_by_you"));
               }
            }
         }

      }

      return false;
   }

   /**
    * Change user primary number
    *
    * @param array $data
    * @return User
    * @throws InvalidPasswordException
    * @throws TelephoneAlreadyExistException
    */
   public static function changePrimaryPhoneNumber(array $data): User
   {
      /** @var User $user */
      $user = auth()->user();

      //Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();

      try {
         //Check phone number is unique and is not already in use by the current user
         self::checkPhoneNumberIsUniqueAndNotUsedByCurrentUser($data["telephone"]);

         //Get raw country from phone number
         $rawCountry = Country::getCountryInfo($data["telephone"]);

         // TODO ensure response is not null
         //Get country model
         $countryModel = Country::findByCountryCode($rawCountry->{"country_code"});

         // Update user phone number and country
         $user->updateService([
            "telephone" => $data["telephone"],
            "country_id" => $countryModel->{"id"},
         ]);

         //Update user telephone
         $user->{"userPrimaryNumber"}->updateService([
            "telephone" => $data["telephone"],
         ]);

         DB::commit();
         $result = $user->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::changePhoneNumber');
         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Change user primary email
    *
    * @param array $data
    * @return User
    * @throws EmailAlreadyExistException
    * @throws InvalidPasswordException
    */
   public static function changePrimaryEmail(array $data): User
   {
      /** @var User $user */
      $user = auth()->user();

      //Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();
      try {
         //Check phone number is unique and is not already in use by the current user
         self::checkEmailIsUniqueAndNotUsedByCurrentUser($data["email"]);

         //Update user email
         $user->updateService([
            "email" => $data["email"],
         ]);

         //Update user email
         $user->{"userPrimaryEmail"}->updateService([
            "email" => $data["email"],
         ]);

         DB::commit();
         $result = $user->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::changeEmail');
         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Change user password
    *
    * @throws PasswordAlreadyUsedException
    * @throws InvalidPasswordException
    * @throws PasswordConfirmationException
    * @throws Exception
    */
   public function changePassword(array $data): User
   {
      $user = $this;
      $user->checkUserPassword($data["old_password"]);

      // Get user password change histories
      $userPasswordChangeHistories = $user->{"userPasswordChangeHistory"};

      // Check password confirmation is valid
      $passwordIsValid = strcmp($data["new_password"], $data["confirm_password"]);
      if ($passwordIsValid != 0) {
         throw new PasswordConfirmationException(__("errors.password_confirmation_incorrect"));
      }

      foreach ($userPasswordChangeHistories as $userPasswordChangeHistory) {
         if ($userPasswordChangeHistory->checkIfNewPassedAlreadyUsed($data["new_password"])) {
            throw new PasswordAlreadyUsedException(__("errors.password_already_used"));
         }
      }

      DB::beginTransaction();

      try {
         // Update user password
         $user->updateService(["password" => Hash::make($data["new_password"])]);

         // Get last user password change history
         /** @var PasswordChangeHistory $lastPasswordChangeHistory */
         $lastPasswordChangeHistory = PasswordChangeHistory::getLastChangeHistory($user->{"id"});

         // Update last password change history to inactive
         $lastPasswordChangeHistory?->updateService([
            "status" => PasswordAndWalletPinChangeStatusEnum::inactive->value,
         ]);

         // Create new user password change history
         PasswordChangeHistory::store([
            "old_password" => Hash::make($data["old_password"]),
            "new_password" => Hash::make($data["new_password"]),
            "status" => PasswordAndWalletPinChangeStatusEnum::active->value,
            "user_id" => $user->{"id"},
         ]);

         DB::commit();
         $result = $user->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::changePassword');

         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Delete user account
    *
    * @param array $data
    * @return Builder|Model
    * @throws InvalidPasswordException
    * @throws Exception
    */
   public function deleteAccount(array $data): Model|Builder
   {
      $user = $this;

      // Verify user password
      $user->checkUserPassword($data["password"]);

      DB::beginTransaction();

      try {
         // Create delete account line
         $deletedAccount = DeletedAccount::store([
            "reason_for_deleting" => $data["reason_for_deleting"] ?? null,
            "data_exported_at" => $data["export_data"] === GenericStatusEnum::enable->value ? now() : null,
            "user_id" => $user->{"id"},
         ]);

         // Delete user account
         $user->delete();
         $user->userDetail()->delete();
         $user->business()->delete();
         $user->otpVerifications()->delete();
         $user->userTelephones()->delete();
         $user->userEmails()->delete();

         DB::commit();
         $result = $deletedAccount->refresh();
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::deleteAccount');

         DB::rollBack();
         throw $exception;
      }

      return $result;
   }

   /**
    * Discover phone contacts
    *
    * @param array $data
    * @return array
    */
   public static function discover(array $data): array
   {
      /** @var User $authUser */
      $authUser = auth()->user();

      $dialCode = $authUser?->{'country'}?->{'dial_code'};
      $contactUsers = [];

      try {
         foreach ($data as $contact) {
            // Compute phones ...
            if (isset($contact['phones']) && is_array($contact['phones'])) {
               $phones = [];
               foreach ($contact['phones'] as $phone) {
                  if (filled($phone)) {
                     $parseResult = parse_phone_number($phone);

                     if ($parseResult === null) {
                        if ($phone !== null) {
                           $phones[$phone] = sprintf('%s%s', $dialCode, remove_space($phone));
                        }
                     } else {
                        $phones[$phone] = sprintf(
                           '+%s%s',
                           $parseResult->getCountryCode(),
                           $parseResult->getNationalNumber()
                        );
                     }
                  }
               }

               // Delete duplication
               $phones = array_unique($phones);
               foreach ($phones as $key => $phone) {
                  // Search ...
                  $user = User::query()->where('telephone', $phone)->first();
                  $isFollower = $user !== null && $user->isFollower($authUser->{"id"});
                  $isFollowingBack = $user !== null && $authUser->isFollower($user->{"id"});

                  $contactUsers[] = array(
                     ...$contact,
                     'user' => $user?->load(['userDetail', 'country', 'business.businessCategory']),
                     'is_follower' => $isFollower ? config("torryme.constants.default_increment_or_decrement") : config("torryme.constants.default_zero_number"),
                     'is_following' => $isFollowingBack ? config("torryme.constants.default_increment_or_decrement") : config("torryme.constants.default_zero_number"),
                     'phones' => [$key],
                  );
               }
            }
         }
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserService::discover');
      }

      return $contactUsers;
   }

   /**
    * Check if is follower
    *
    * @param int $followingUserId
    * @return bool
    */
   public function isFollower(int $followingUserId): bool
   {
      $user = $this;

      return Follower::query()
         ->where("follower_user_id", $user->{'id'})
         ->where("following_user_id", $followingUserId)
         ->exists();
   }

   /**
    * Can ate product
    *
    * @param int $productId
    * @return bool
    */
   public function canRateProduct(int $productId): bool
   {
      $hasPurchasedProduct = Order::checkProductPurchase($this->{"id"}, $productId);

      return $hasPurchasedProduct !== null;
   }

   /**
    * WHo can see followers list
    *
    * @param User $visitedProfile
    * @return bool
    */
   public function whoCanSeeFollowerList(User $visitedProfile): bool
   {
      /** @var User $authUser */
      $authUser = auth()->user();

      $canSeeFollowerList = true;

      // Get visited profile account privacy setting
      $visitedProfilePrivacySetting = $visitedProfile->{"accountPrivacySetting"};

      if ($visitedProfilePrivacySetting->{"enable_privacy"} == GenericStatusEnum::enable->value) {
         if ($visitedProfilePrivacySetting->{"who_can_see_follower_list"} == AccountPrivacyEnum::everyOne->value || $visitedProfilePrivacySetting->{"who_can_see_follower_list"} == AccountPrivacyEnum::phoneContacts->value) {
            $canSeeFollowerList = true;
         } elseif ($visitedProfilePrivacySetting->{"who_can_see_follower_list"} == AccountPrivacyEnum::noOne->value) {
            $canSeeFollowerList = false;
         } elseif ($visitedProfilePrivacySetting->{"who_can_see_follower_list"} == AccountPrivacyEnum::followers->value) {
            $canSeeFollowerList = $authUser->isFollower($visitedProfile->{"id"});
         }
      }

      return $canSeeFollowerList;
   }

   /**
    * Get followers
    *
    * @param int|null $pageNumber
    * @return LengthAwarePaginator
    */
   public function getFollowers(int $pageNumber = null): LengthAwarePaginator
   {
      return Follower::getFollowers($this, $pageNumber);
   }

   /**
    * All accounts followed by the user
    *
    * @param int|null $pageNumber
    * @return LengthAwarePaginator
    */
   public function getFollowings(int $pageNumber = null): LengthAwarePaginator
   {
      return Follower::getFollowings($this, $pageNumber);
   }

   /**
    * Can mention another user ?
    *
    * @param User $user
    * @return bool
    */
   public function canMentionAnother(User $user): bool
   {
      $accountPrivacySetting = $user->{'accountPrivacySetting'};

      if ($accountPrivacySetting->{"enable_privacy"} == GenericStatusEnum::enable->value) {
         if ($accountPrivacySetting->{"who_can_mention"} == AccountPrivacyEnum::everyOne->value || $accountPrivacySetting->{"who_can_mention"} == AccountPrivacyEnum::phoneContacts->value) {
            return true;
         } else if ($accountPrivacySetting->{"who_can_mention"} == AccountPrivacyEnum::followers->value) {
            return $this->isFollower($user->{"id"});
         }
      }

      return true;
   }

   /**
    * User has enable two auth factor ?
    *
    * @return bool
    */
   public function twoFactorAuthIsEnable(): bool
   {
      $accountPrivacySetting = $this->{'accountPrivacySetting'};

      return $accountPrivacySetting?->{"enable_2factor_auth"} === GenericStatusEnum::enable->value;
   }

   /**
    * User suggestions
    *
    * @param int|null $pageNumber
    * @return array
    */
   public function userSuggestions(int $pageNumber = null): array
   {
      return Follower::userSuggestions($this, $pageNumber);
   }

   /**
    * Is emitter
    *
    * @param RealtimeMessage $message
    * @return bool
    */
   public function isEmitter(RealtimeMessage $message): bool
   {
      return $message->{'emitter_participant_id'} === $this->{'participant'}->{'id'};
   }

   /**
    * Is receiver
    *
    * @param RealtimeMessage $message
    * @return bool
    */
   public function isReceiver(RealtimeMessage $message): bool
   {
      return $message->{'receiver_participant_id'} === $this->{'participant'}->{'id'};
   }

   /**
    * Get verified emails
    *
    * @return Collection
    */
   public function verifiedEmails(): Collection
   {
      return $this->userEmails()->whereNotNull('verified_at')->get();
   }
}
