<?php

namespace App\Services;

use App\Models\SeasonEpisode;
use App\Models\Series;
use App\Models\SeriesSeason;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;

trait SeriesSeasonService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SeriesSeason::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return SeriesSeason::query()->find($id);
    }

    /**
     * Find season by occurrence
     *
     * @param int $occurrence
     * @param int $seriesId
     * @return Model|Collection|Builder|array|null
     */
    public static function findByOccurrence(int $occurrence, int $seriesId): Model|Collection|Builder|array|null
    {
        return
            SeriesSeason::query()
                ->where('occurrence', $occurrence)
                ->where('series_id', $seriesId)
                ->first();
    }

    /**
     * Add new season
     *
     * @param array $data
     * @param Series $series
     * @return Builder|Model
     */
    public static function createSeason(array $data, Series $series): Model|Builder
    {
        return self::store([
            'designation' => $data['season_title'] ?? null,
            'description' => $data['season_description'] ?? null,
            'occurrence' => $data['season_occurrence'],
            'downloadable_price' => isset($data['season_price']) && (float) $data['season_price'] >= config("torryme.constants.default_zero_number") ? (float) $data['season_price'] : config('torryme.constants.default_price'),
            'downloadable_price_per_episode' => isset($data['episode_season_price']) && (float) $data['episode_season_price'] >= config("torryme.constants.default_zero_number") ? (float) $data['episode_season_price'] : config('torryme.constants.default_price'),
            'price_currency_id' => $series->{'price_currency_id'},
            'series_id' => $series->{'id'}
        ]);
    }

    /**
     *  Add new season episode
     *
     * @param UploadedFile $episodeFile
     * @param array $data
     * @return Model|Builder|null
     * @throws \Exception
     */
    public function addEpisode(UploadedFile $episodeFile, array $data): Model|Builder|null
    {
        return SeasonEpisode::addNewEpisodeForSeason($this, $episodeFile, $data);
    }

    /**
     * Total number of season in the series
     *
     * @param Series $series
     * @return int
     */
    public static function totalSeriesSeason(Series $series): int
    {
        return
            SeriesSeason::query()
                ->whereHas('series', function(Builder $builder) use($series) {
                    $builder->where('id', $series->{'id'});
                })
                ->count();
    }

    /**
     * Total number of episodes in the series season
     *
     * @return int
     */
    public function totalEpisode(): int
    {
        return SeasonEpisode::totalSeriesSeasonEpisode($this);
    }

    /**
     * Compute sample series seasons attributes
     *
     * @param Series $series
     * @param Collection $seasons
     * @param bool $withEpisodes
     * @return array
     */
    public static function computeSampleDetails(Series $series, Collection $seasons, bool $withEpisodes = false): array
    {
        $result = [];

        foreach ($seasons as $season) {
            $episodeData = array(
                'cover_url' => $season->{'cover_url'},
                'occurrence' => $season->{'occurrence'},
                'designation' => $season->{'designation'},
                'description' => $season->{'description'},
                'total_episode' => $season->totalEpisode(),
                'size' => $season->totalSize(),
                "downloadable_price" => $season->{'downloadable_price'},
                "downloadable_price_per_episode" => $season->{'downloadable_price_per_episode'},
                "price_currency" => $season->{'currency'}?->{'code'} ?? $series->{'currency'}->{'code'},
            );

            if($withEpisodes) {
                $episodeData['episodes'] = SeasonEpisode::computeSampleDetails(
                    $season->{'seasonEpisodes'}
                );
            }

            $result[] = $episodeData;
        }

        return $result;
    }
}
