<?php

namespace App\Services;

use App\Enums\AttachmentTypeEnum;
use App\Enums\DocumentTypeEnum;
use App\Models\PostComment;
use App\Models\PostCommentAttachment;
use App\Models\Post;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

trait PostCommentsService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return PostComment::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Create post comment
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function commentPost(array $data): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();

        DB::beginTransaction();
        try {
            /** @var PostComment $postComment */

            $postComment = self::store([
                "comment" => $data["comment"],
                "post_id" => $data["post_id"],
                'user_id' => $user->{'id'},
                "post_comment_id" => null,
            ]);

            // Add post comments
            $postComment->{"currentPost"}->addNumberOfPostComment();

            self::handleCommentAttachment($data, $user, $postComment, $data["post_id"]);

            DB::commit();
            $result = $postComment;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'PostCommentsService::commentPost');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return PostComment::query()->find($id);
    }

    /**
     * Reply post comment
     *
     * @param array $data
     * @return PostComment|null
     */
    public function commentReply(array $data): ?PostComment
    {
        /** @var User $user */
        $user = auth()->user();

        DB::beginTransaction();

        try {
            $commentToReply = $this;

            /** @var PostComment $reply */
            $reply = self::store([
                "comment" => $data["comment"],
                'user_id' => $user->{'id'},
                "post_id" => $commentToReply->{"post_id"},
                "post_comment_id" => $commentToReply->{"id"},
            ]);

            // Add post comments
            $reply->{"currentPost"}->addNumberOfPostComment();

            self::handleCommentAttachment($data, $user, $reply, $reply->{"post_id"});

            DB::commit();
            $result = $reply->refresh();
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'PostCommentsService::commentReply');
            DB::rollBack();

            $result = null;
        }

        return $result;
    }

    /**
     * Handle comment attachments
     *
     * @param array $data
     * @param User $user
     * @param PostComment $postComment
     * @param int $postId
     * @return void
     */
    public static function handleCommentAttachment(array $data, User $user, PostComment $postComment, int $postId): void
    {
        if (isset($data["attachments"])) {
            $attachmentName = DocumentTypeEnum::commentAttachment->value;
            $userCurrentDocPath = sprintf(config('torryme.paths.docs'), User::$prefixDir . $user->{'id'});

            // Get post directory
            $postPath = sprintf($userCurrentDocPath . '/%s', Post::$prefixDir . $postId);
            // Get comment directory
            $commentPath = sprintf($postPath . '/%s', PostComment::$prefixDir . $postComment->{"id"});

            //Upload comment attachment
            $path = upload_image($attachmentName, $commentPath, $data["attachments"]);

            // TODO Delete comment attachment in case of error
            PostCommentAttachment::store([
                "path" => $path,
                "type" => AttachmentTypeEnum::image->value,
                "post_comment_id" => $postComment->{"id"},
            ]);
        }
    }

    /**
     * Get post comments
     *
     * @param int $postId
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function getCommentPost(int $postId, int $pageNumber = null): LengthAwarePaginator
    {
        $queryBuilder =
            PostComment::query()
                ->where("post_id", $postId)
                ->whereNull("post_comment_id")
                ->orderByDesc('created_at');

        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Compute details for comment
     *
     * @return array
     */
    public function computeCommentDetails(): array
    {
        $comment = $this;
        $userCommenter = $comment->{"userCommenter"};

        return array(
            "id" => $comment->{"id"},
            "comment" => $comment->{"comment"},
            "user_commenter" => $userCommenter->load('userDetail'),
            "comment_attachments" => $comment->{"commentAttachment"},

            "created_at" => Carbon::parse($comment->{'created_at'})->utc(),
        );
    }

    /**
     * Get comment answers
     *
     * @param int|null $page
     * @return LengthAwarePaginator|Builder[]|Collection
     */
    public function getAnswers(int $page = null): Collection|LengthAwarePaginator|array
    {
        $comment = $this;

        $queryBuilder =
            PostComment::query()
                ->where("post_comment_id", $comment->{'id'})
                ->orderByDesc('created_at');

        if($page !== null) {
            $paginator =
                $queryBuilder->paginate(
                    perPage: config("torryme.constants.items_per_page"),
                    page: $page
                );

            $result = array();
            foreach ($paginator as $comment) {
                $result[] = $comment->computeCommentDetails();
            }

            return default_paginator_format(
                $paginator->lastPage(),
                $paginator->total(),
                $paginator->currentPage(),
                "comments",
                $result,
            );
        }

        return $queryBuilder->get();
    }
}
