<?php

namespace App\Services;

use App\Exceptions\InvalidFollowException;
use App\Models\Business;
use App\Models\Follower;
use App\Models\User;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use ProtoneMedia\LaravelCrossEloquentSearch\Search;

trait FollowerService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Follower::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * CHeck is not self following
     *
     * @param int $followerId
     * @param int $followingId
     * @throws InvalidFollowException
     */
    protected static function isSelfFollowing(int $followerId, int $followingId): void
    {
        if ($followerId === $followingId) {
            throw new InvalidFollowException(__("errors.invalid_follow_action"));
        }
    }

    /**
     * We try to do instant search of a user's followers
     *
     * @param User $user
     * @param string $query
     * @param int|null $pageNumber
     * @return array|Collection
     */
    public static function lookingUsersToIdentify(User $user, string $query, int $pageNumber = null): array|Collection
    {

       if(filled($query)) {
          $queryBuilder =
             Search::add(Follower::query()->with('follower')->where('following_user_id', $user->{'id'}), [
                'follower.user_name', 'follower.userDetail.first_name', 'follower.userDetail.last_name'
             ])
                ->add(Follower::query()->with('following')->where('follower_user_id', $user->{'id'}), [
                   'following.user_name', 'following.userDetail.first_name', 'following.userDetail.last_name'
                ])
                ->beginWithWildcard()
                ->search($query);

          $userResultsArray = array();
          foreach ($queryBuilder as $result) {
             if ($result->{"follower_user_id"} == $user->{"id"}) {
                $result = $result->{"following"};
             } else {
                $result = $result->{"follower"};
             }

             $userResultsArray[] = $result;
          }

          $resultsBeforePrivacy = array_unique($userResultsArray);

          $userIdsAfterPrivacy = array();
          foreach ($resultsBeforePrivacy as $userCanBeMentioned) {
             if ($user->canMentionAnother($userCanBeMentioned)) {
                $userIdsAfterPrivacy[] = $userCanBeMentioned->{"id"};
             }
          }

          $mentionUsers =
             User::query()
                ->whereIn("id", $userIdsAfterPrivacy)
                ->paginate(
                   perPage: config("torryme.constants.items_per_page"),
                   page: $pageNumber
                );

          $mentionUsersResults = array();
          foreach ($mentionUsers as $user) {
             $mentionUsersResults[] = $user->load(['userDetail', 'business.businessCategory']);
          }

          return default_paginator_format(
             $mentionUsers->lastPage(),
             $mentionUsers->total(),
             $mentionUsers->currentPage(),
             'mentioned_users',
             $mentionUsersResults,
          );

       }

        return default_paginator_format(
            1,
            0,
            1,
            'mentioned_users',
            [],
        );
    }

    /**
     * Follow or unfollow user
     *
     * @param int $followingUserId
     * @return mixed
     * @throws Exception
     */
    public static function followUser(int $followingUserId): mixed
    {
        try {
            /** @var User $user */
            $user = auth()->user();

            // Check is not self following
            self::isSelfFollowing($user->{"id"}, $followingUserId);

            // Check if is already a follower
            $alreadyFollower = $user->isFollower($followingUserId);
            if ($alreadyFollower) {
                $action = self::unfollowUser($followingUserId);
            } else {
                $action = Follower::store([
                    "follower_user_id" => $user->{"id"},
                    "following_user_id" => $followingUserId,
                ]);
            }

            $result = $action;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubscriberService::followUser');
            throw $exception;
        }

        return $result;
    }

    /**
     * Unfollow user
     *
     * @param int $followingUserId
     * @return mixed
     */
    public static function unfollowUser(int $followingUserId): mixed
    {
        /** @var User $user */
        $user = auth()->user();

        return
            Follower::query()
                ->where("follower_user_id", $user->{"id"})
                ->where("following_user_id", $followingUserId)
                ->delete();
    }

    /**
     * Get user followers
     *
     * @param User $user
     * @param int|null $pageNumber
     * @return LengthAwarePaginator|Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public static function getFollowers(User $user, int $pageNumber = null): \Illuminate\Database\Eloquent\Collection|LengthAwarePaginator|array
    {
        $queryBuilder =
            Follower::query()
                ->with(["follower.userDetail"])
                ->where("following_user_id", $user->{"id"})
                ->orderByDesc('created_at');

        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Search followers
     *
     * @param User $user
     * @param string $query
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function searchFollowers(User $user, string $query, int $pageNumber = null): LengthAwarePaginator
    {
        $queryBuilder =
            Follower::query()
                ->where('following_user_id', $user->{'id'})
                ->whereHas('follower', function (Builder $builder) use ($query) {
                    $builder
                        ->where('user_name', 'LIKE', '%' . $query . '%')
                        ->orWhereHas('business', function (Builder $builder) use ($query) {
                            $builder->where('designation', 'LIKE', '%' . $query . '%');
                        })
                        ->orWhereHas('userDetail', function (Builder $builder) use ($query) {
                            $builder->where(
                                DB::raw("CONCAT(first_name, ' ', last_name)"), 'LIKE', "%" . $query . "%"
                            );
                        });
                });

        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Users followed
     *
     * @param User $user
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function getFollowings(User $user, int $pageNumber = null): LengthAwarePaginator
    {
        $queryBuilder =
            Follower::query()
                ->with(["following.userDetail"])
                ->where("follower_user_id", $user->{"id"})
                ->orderByDesc('created_at');

        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Search followings
     *
     * @param User $user
     * @param string $query
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function searchFollowings(User $user, string $query, int $pageNumber = null): LengthAwarePaginator
    {

        $queryBuilder =
            Follower::query()
                ->where('follower_user_id', $user->{'id'})
                ->whereHas('following', function (Builder $builder) use ($query) {
                    $builder
                        ->where('user_name', 'LIKE', '%' . $query . '%')
                        ->orWhereHas('business', function (Builder $builder) use ($query) {
                            $builder->where('designation', 'LIKE', '%' . $query . '%');
                        })
                        ->orWhereHas('userDetail', function (Builder $builder) use ($query) {
                            $builder->where(
                                DB::raw("CONCAT(first_name, ' ', last_name)"), 'LIKE', "%" . $query . "%"
                            );
                        });
                });


        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Delete follower
     *
     * @param int $followerUserId
     * @return mixed
     */
    public static function deleteFollower(int $followerUserId): mixed
    {
        /** @var User $user */
        $user = auth()->user();

        return Follower::query()->where("following_user_id", $user->{"id"})->where("follower_user_id", $followerUserId)->delete();
    }

    /**
     * Get number of user's followers
     *
     * @param Int $userId
     * @return int
     */
    public static function getNumberOfFollowers(int $userId): int
    {
        return Follower::query()->where("following_user_id", $userId)->count();
    }

    /**
     * Get number of user's following
     * @param int $userId
     * @return int
     */
    public static function getNumberOfFollowings(int $userId): int
    {
        return Follower::query()->where("follower_user_id", $userId)->count();
    }

    /**
     * Serialize data for follower query search
     *
     * @param $followers
     * @return array
     */
    public static function serializeFollowers($followers): array
    {
        $followersArray = array();

        /** @var User $authUser */
        $authUser = auth()->user();

        foreach ($followers as $follower) {
            $isFollowingBack = $authUser->isFollower($follower->{"follower_user_id"});
            $follower = $follower->{"follower"}->load(['userDetail', 'business.businessCategory']);

            $follower['is_following_back'] = $isFollowingBack ? config("torryme.constants.default_one_number") : config("torryme.constants.default_zero_number");

            $followersArray[] = $follower;
        }

        return $followersArray;
    }

    /**
     * Serialize data for following query search
     *
     * @param $followings
     * @return array
     */
    public static function serializeFollowings($followings): array
    {
        $followingArray = array();

        foreach ($followings as $following) {
            $following = $following->{"following"}->load(['userDetail', 'business']);
            $followingArray[] = $following;
        }

        return $followingArray;
    }

    /**
     * Check if is follower
     *
     * @param int $followingUserId
     * @return bool
     */
    public function isFollower(int $followingUserId): bool
    {
        /** @var User $user */
        $authUser = $this;

        return
            Follower::query()
                ->where("following_user_id", $followingUserId)
                ->where("follower_user_id", $user->{"id"})
                ->exists();
    }

    /**
     * User suggestions
     *
     * @param User $user
     * @param int|null $pageNumber
     * @return array
     */
    public static function userSuggestions(User $user, int $pageNumber = null): array
    {
        /** @var User $authUser */
        $authUser = auth()->user();

        /** @var Business $business */
        $business = $user->{"business"};
        $businessOwners = array();

        if ($business != null) {
            // Get 10 business with the same business category
            $businessesWithSameCategory = $business->getBusinessByCategory();

            // Get corresponding business owners
            foreach ($businessesWithSameCategory as $sameCategoryBusiness) {
                $businessOwners[] = $sameCategoryBusiness->{"ownerUser"};
            }
        }

        // Get followers and followings of user
        $queryBuilder =
            Follower::query()
                ->with(["follower", "following"])
                ->where("following_user_id", $user->{"id"})
                ->orWhere("follower_user_id", $user->{"id"})
                ->get();

        $userResultsArray = array();
        foreach ($queryBuilder as $result) {
            $userResultsArray[] = $result->{"follower_user_id"} === $user->{"id"} ? $result->{"following"} : $result->{"follower"};
        }

        // Merge business and follower/following results
        $mergedUsers = array_merge($businessOwners, $userResultsArray);

        // Filter array to remove all duplicates
        $filteredResults = array_unique($mergedUsers);

        $suggestedUserArray = array();
        foreach ($filteredResults as $suggestedUser) {
            $suggestedUserArray[] = array(
                "id" => $suggestedUser->{"id"},
                "number_of_followers" => self::getNumberOfFollowers($suggestedUser->{"id"}),
                "number_of_followings" => self::getNumberOfFollowings($suggestedUser->{"id"}),
            );
        }

        $numberOfFollowers = array_column($suggestedUserArray, 'number_of_followers');
        $numberOfFollowings = array_column($suggestedUserArray, 'number_of_followings');

        // Sort suggested users according to number of followers and followings
        array_multisort($numberOfFollowers, SORT_DESC, $numberOfFollowings, SORT_DESC, $suggestedUserArray);

        // Get suggested user IDs after sorting
        $sortedUserIdArray = array_column($suggestedUserArray, 'id');

        // Convert sorted suggested user IDs to string
        $sortedIds = implode(',', $sortedUserIdArray);

        $paginatedUsers =
            User::query()
                ->whereIn("id", $sortedUserIdArray)
                ->whereNot("id", $authUser->{"id"})
                ->orderByRaw("FIELD(id, {$sortedIds})")
                ->paginate(
                    perPage: config("torryme.constants.items_per_page"),
                    page: $pageNumber
                );

        $userResults = array();
        foreach ($paginatedUsers as $userSuggestion) {
            $userResults[] = $userSuggestion->load(['userDetail', 'business.businessCategory']);
        }

        return default_paginator_format(
            $paginatedUsers->lastPage(),
            $paginatedUsers->total(),
            $paginatedUsers->currentPage(),
            'suggested_users',
            $userResults,
        );
    }
}
