<?php

namespace App\Services;

use App\Enums\ExternalAccountStatusEnum;
use App\Models\ExternalAccount;
use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait ExternalAccountService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return ExternalAccount::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return ExternalAccount::query()->find($id);
    }

    /**
     * Find phone number by gateway
     *
     * @param string $gateWay
     * @param string $direction
     * @param string $telephone
     * @return Model|Builder|null
     */
    public static function findPhoneNumberByGateway(string $gateWay, string $direction, string $telephone): Model|Builder|null
    {
        return ExternalAccount::query()->where("payment_gateway", $gateWay)
            ->where("operation_direction", $direction)
            ->where("telephone", $telephone)->first();
    }

    /**
     * Find email by gateway
     *
     * @param string $gateWay
     * @param string $direction
     * @param string $email
     * @return Model|Builder|null
     */
    public static function findEmailByGateway(string $gateWay, string $direction, string $email): Model|Builder|null
    {
        return ExternalAccount::query()->where("payment_gateway", $gateWay)
            ->where("operation_direction", $direction)
            ->where("email", $email)->first();
    }

    /**
     * Find card number by gateway
     *
     * @param string $gateWay
     * @param string $direction
     * @param string $cardNumber
     * @return Model|Builder|null
     */
    public static function findCardNumberByGateway(string $gateWay, string $direction, string $cardNumber): Model|Builder|null
    {
        return ExternalAccount::query()->where("payment_gateway", $gateWay)
            ->where("operation_direction", $direction)
            ->where("card_number", $cardNumber)->first();
    }

    /**
     * Configure account
     *
     * @param array $data
     * @return Builder|Model|null
     * @throws Exception
     */
    public static function configureAccount(array $data): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();

        //Get user wallet
        $wallet = $user->{"wallet"};

        //Check current pin match
        $wallet->verifyWalletPin($data["wallet_pin"]);

        try {
            $newExternalAccount = null;
            $existingExternalAccount = null;

            //Get operational direction
            $operationDirection = get_external_account_direction($data["operation_direction"]);

            if (isset($data["telephone"])) {
                //Check for existing external account with given telephone number
                $existingExternalAccount = self::findPhoneNumberByGateway($data["payment_gateway"], $operationDirection, $data["telephone"]);

                if ($existingExternalAccount == null) {
                    $newExternalAccount = self::store([
                        'telephone' => $data["telephone"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }
            }

            if (isset($data["email"])) {
                //Check for existing external account with given email
                $existingExternalAccount = self::findEmailByGateway($data["payment_gateway"], $operationDirection, $data["email"]);

                if ($existingExternalAccount == null) {
                    $newExternalAccount = self::store([
                        'email' => $data["email"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }
            }

            if (isset($data["card_number"])) {
                //Check for existing external account with given card number
                $existingExternalAccount = self::findCardNumberByGateway($data["payment_gateway"], $operationDirection, $data["card_number"]);

                if ($existingExternalAccount == null) {
                    $newExternalAccount = ExternalAccount::store([
                        'card_number' => $data["card_number"],
                        'card_pin' => $data["card_pin"],
                        'card_expiry_date' => $data["card_expiry_date"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }
            }

            //TODO Optimise code to update number, email, card number when creating account using insomnia

            $result = $existingExternalAccount != null ? $existingExternalAccount->refresh() : $newExternalAccount->refresh();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ExternalAccountService::configureAccount');

            throw  $exception;
        }

        return $result;
    }

    /**
     * Edit account
     *
     * @param array $data
     * @return Model|Collection|Builder|array|null
     * @throws Exception
     */
    public static function editAccount(array $data): Model|Collection|Builder|array|null
    {
        /** @var User $user */
        $user = auth()->user();

        //Get user wallet
        $wallet = $user->{"wallet"};

        //Check current pin match
        $wallet->verifyWalletPin($data["wallet_pin"]);

        try {
            $existingExternalAccount = null;
            $newExternalAccount = null;

            $activeExternalAccount = self::findById($data["external_account_id"]);

            //Get operational direction
            $operationDirection = get_external_account_direction($data["operation_direction"]);

            if (isset($data["telephone"])) {

                //Check for existing external account with given telephone number
                $existingExternalAccount = self::findPhoneNumberByGateway($data["payment_gateway"], $operationDirection, $data["telephone"]);

                if ($existingExternalAccount != null) {
                    //Check if request phone number equal to existing phone number
                    if ($activeExternalAccount->{"telephone"} != $existingExternalAccount->{"telephone"}) {

                        //Deactivate current account number
                        $activeExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::inactive->value,
                        ]);

                        //Activate old account number
                        $existingExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::active->value,
                        ]);
                    } else {
                        $activeExternalAccount->update([
                            'name' => $data["name"],
                        ]);
                    }
                } else {
                    //Deactivate current account number
                    $activeExternalAccount->update([
                        'status' => ExternalAccountStatusEnum::inactive->value,
                    ]);

                    //Create new account
                    $newExternalAccount = self::store([
                        'telephone' => $data["telephone"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }
            }

            if (isset($data["email"])) {
                //Check for existing external account with given email
                $existingExternalAccount = self::findEmailByGateway($data["payment_gateway"], $operationDirection, $data["email"]);

                if ($existingExternalAccount != null) {
                    //Check if request email equal to existing email
                    if ($activeExternalAccount->{"email"} != $existingExternalAccount->{"email"}) {

                        //Deactivate current email
                        $activeExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::inactive->value,
                        ]);

                        //Activate old email
                        $existingExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::active->value,
                        ]);
                    } else {
                        $activeExternalAccount->update([
                            'name' => $data["name"],
                        ]);
                    }
                } else {
                    //Deactivate current email
                    $activeExternalAccount->update([
                        'status' => ExternalAccountStatusEnum::inactive->value,
                    ]);

                    //Create new account
                    $newExternalAccount = self::store([
                        'email' => $data["email"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }
            }

            if (isset($data["card_number"])) {
                //Check for existing external account with given card number
                $existingExternalAccount = self::findCardNumberByGateway($data["payment_gateway"], $operationDirection, $data["card_number"]);

                if ($existingExternalAccount != null) {
                    //Check if request email equal to existing email
                    if ($activeExternalAccount->{"card_number"} != $existingExternalAccount->{"card_number"}) {

                        //Deactivate current given card number
                        $activeExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::inactive->value,
                        ]);

                        //Activate old given card number
                        $existingExternalAccount->update([
                            'status' => ExternalAccountStatusEnum::active->value,
                        ]);
                    } else {
                        $activeExternalAccount->update([
                            'card_pin' => $data["card_pin"],
                            'card_expiry_date' => $data["card_expiry_date"],
                            'name' => $data["name"],
                        ]);
                    }
                } else {
                    //Deactivate current given card number
                    $activeExternalAccount->update([
                        'status' => ExternalAccountStatusEnum::inactive->value,
                    ]);

                    //Create new account
                    $newExternalAccount = self::store([
                        'card_number' => $data["card_number"],
                        'card_pin' => $data["card_pin"],
                        'card_expiry_date' => $data["card_expiry_date"],
                        'status' => ExternalAccountStatusEnum::active->value,
                        "payment_gateway" => get_payment_gateway($data["payment_gateway"]),
                        'operation_direction' => $operationDirection,
                        'name' => $data["name"],
                        'user_id' => $user->{"id"},
                    ]);
                }

            }

            $result = $existingExternalAccount != null ? $existingExternalAccount->refresh() : $newExternalAccount->refresh();

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ExternalAccountService::editAccount');

            throw  $exception;
        }

        return $result;
    }

    /**
     * Get user accounts
     *
     * @param string $direction
     * @return Builder[]|Collection|null
     */
    public static function userAccounts(string $direction): Collection|array|null
    {
        try {
            /** @var User $user */
            $user = auth()->user();

            $status = ExternalAccountStatusEnum::active->value;

            $externalAccount = ExternalAccount::query()->where("operation_direction", $direction)
                ->where("user_id", $user->{"id"})
                ->where("status", $status)
                ->get();

            $result = $externalAccount;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'ExternalAccountService::userAccounts');

            $result = null;
        }

        return $result;
    }
}
