<?php

namespace App\Services;

use App\Models\Dispute;
use App\Models\DisputeMessage;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait DisputeMessageService
{
   /**
    * Create new
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return DisputeMessage::query()->create($data);
   }

   /**
    * Serialize for channel interaction
    *
    * @return array
    */
   public function serializeForChannel(): array
   {
      $message = $this;

      return array(
         'uuid' => $message->{'uuid'},
         'dispute_uuid' => $message->{'dispute'}->{'uuid'},
         'message' => $message->{'message'},
         'sent_at' => $message->{'sent_at'},
         'emitter' => $message->{'emitter'}->load(['user.userDetail', 'user.business']),
      );
   }

   /**
    * New message
    *
    * @param array $data
    * @return array|null
    */
   public static function newMessage(array $data): ?array
   {
      DB::beginTransaction();

      /** @var User $user */
      $user = auth()->user();

      try {
         // Dispute ...
         /** @var Dispute $dispute */
         $dispute = Dispute::findByUuid($data['dispute_uuid']);

         // Emitter participant ...
         $emitterParticipant = $user->{'disputeParticipant'};

         // Create message
         /** @var DisputeMessage $message */
         $message = self::store([
            'uuid' => generate_uuid('dispute_messages'),
            'message' => $data['message'],
            'sent_at' => now(),

            'dispute_id' => $dispute->{'id'},
            'emitter_participant_id' => $emitterParticipant->{'id'},
            'respond_to_message_id' => null,
         ]);

         DB::commit();

         $message = $message->refresh();

         // Broadcast event to send message ...
         $message->broadcastMessage();

         // Send notification
         $message->sendNewMessageNotification();

         return $message->serializeForChannel();
      } catch (\Exception $exception) {
         DB::rollBack();
         log_debug($exception, 'ChatMessage::newMessage Unable to send message');
      }

      return null;
   }
}
